/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

/////////////////////// StdLib includes
#include <cmath>
#include <map>

/////////////////////// Qt includes
#include <QDateTime>
#include <QDebug>
#include <QFile>
#include <QString>
#include <QJSEngine>

/////////////////////// pappsomspp includes
#include "pappsomspp/core/massspectrum/massspectrum.h"
#include "pappsomspp/core/utils.h"

/////////////////////// Local includes
#include "pappsomspp/core/processing/combiners/mzintegrationparams.h"

/*
int mzIntegrationParamsMetaTypeId =
  qRegisterMetaType<pappso::MzIntegrationParams>("pappso::MzIntegrationParams");
int mzIntegrationParamsPtrMetaTypeId =
  qRegisterMetaType<pappso::MzIntegrationParams *>("pappso::MzIntegrationParams
*");*/

namespace pappso
{

//! Map relating the BinningType to a textual representation
std::map<MzIntegrationParams::BinningType, QString> binningTypeMap{
  {MzIntegrationParams::BinningType::NONE, "NONE"},
  {MzIntegrationParams::BinningType::DATA_BASED, "DATA_BASED"},
  {MzIntegrationParams::BinningType::ARBITRARY, "ARBITRARY"}};

MzIntegrationParams::BinningType
getBinningTypeFromString(const QString &text)
{
  std::map<MzIntegrationParams::MzIntegrationParams::BinningType,
           QString>::const_iterator the_iterator_const =
    std::find_if(
      binningTypeMap.begin(),
      binningTypeMap.end(),
      [text](const std::pair<MzIntegrationParams::BinningType, QString> &pair) {
        return pair.second == text;
      });

  if(the_iterator_const != binningTypeMap.end())
    return the_iterator_const->first;

  return MzIntegrationParams::BinningType::NONE;
}

MzIntegrationParams::MzIntegrationParams(QObject *parent): QObject(parent)
{
}

MzIntegrationParams::MzIntegrationParams(const QString &text, QObject *parent)
  : QObject(parent)
{
  initialize(text);
}

MzIntegrationParams::MzIntegrationParams(
  double min_mz,
  double max_mz,
  MzIntegrationParams::BinningType binning_type,
  pappso::PrecisionPtr bin_size_model,
  int bin_size_divisor,
  int decimal_places,
  bool remove_zero_val_data_points,
  QObject *parent)
  : QObject(parent),
    m_smallestMz(min_mz),
    m_greatestMz(max_mz),
    m_binningType(binning_type),
    m_binSizeModel(bin_size_model),
    m_binSizeDivisor(bin_size_divisor),
    m_decimalPlaces(decimal_places),
    m_removeZeroValDataPoints(remove_zero_val_data_points)
{
  if(m_binSizeModel == nullptr)
    m_binSizeModel = pappso::PrecisionFactory::getPpmInstance(20);

  // qDebug() << "MetaObject?" << this->metaObject()->className();
}

MzIntegrationParams::~MzIntegrationParams()
{
}

MzIntegrationParams *
MzIntegrationParams::clone(QObject *parent) const
{
  MzIntegrationParams *mz_integration_params_p =
    new MzIntegrationParams(m_smallestMz,
                            m_greatestMz,
                            m_binningType,
                            m_binSizeModel,
                            m_binSizeDivisor,
                            m_decimalPlaces,
                            m_removeZeroValDataPoints,
                            parent);

  return mz_integration_params_p;
}

MzIntegrationParams::InitializationResult
MzIntegrationParams::initialize(const QString &text)
{
  // qDebug() << "Initializing MzIntegrationParams using text:" << text;

  InitializationResult initialization_result = InitializationResult::DEFAULT;

  if(text.isEmpty())
    return initialization_result;

  reset();

  // Expected text: "Smallest (first) m/z:294.725158\nGreatest (last)
  // m/z:2055.002453\nDecimal places:-1\nBinning type:2\nBin size model:0.05
  // dalton\nBin size divisor:1\nRemove 0-val data points:1\n"

  // In order to consider that the parameters were correctly read, we need
  // some of the members to effectively be set from the settings values.

  bool binning_type_set            = false;
  bool decimal_places_set          = false;
  bool bin_size_model_set          = false;
  bool bin_size_divisor_set        = false;
  bool remove_zero_data_points_set = false;

  QStringList string_list = text.split("\n");

  for(int iter = 0; iter < string_list.size(); ++iter)
    {
      QString iter_string = string_list.at(iter);

      // qDebug() << "Iterating in string:" << iter_string;

      if(iter_string.contains("Binning type:"))
        {
          setBinningType(
            getBinningTypeFromString(iter_string.split(':').last()));
          binning_type_set = true;
        }
      else if(iter_string.contains("Bin size model:"))
        {
          setBinSizeModel(
            PrecisionFactory::fromString(iter_string.split(':').last()));
          bin_size_model_set = true;
        }
      else if(iter_string.contains("Bin size divisor:"))
        {
          setBinSizeDivisor(iter_string.split(':').last().toInt());
          bin_size_divisor_set = true;
        }
      else if(iter_string.contains("Decimal places:"))
        {
          setDecimalPlaces(iter_string.split(':').last().toInt());
          decimal_places_set = true;
        }
      else if(iter_string.contains("Remove 0-val data points:"))
        {
          setRemoveZeroValDataPoints(iter_string.split(':').last().toInt());
          remove_zero_data_points_set = true;
        }
    }

  // qDebug() << "At this point the initialization is done and this"
  //             " MzIntegrationParams instance is: "
  //          << toString();

  if(binning_type_set)
    initialization_result |= InitializationResult::BINNING_TYPE;
  if(bin_size_model_set)
    initialization_result |= InitializationResult::BIN_SIZE_MODEL;
  if(bin_size_divisor_set)
    initialization_result |= InitializationResult::BIN_SIZE_DIVISOR;
  if(decimal_places_set)
    initialization_result |= InitializationResult::DECIMAL_PLACES;

  if(remove_zero_data_points_set)
    initialization_result |= InitializationResult::REMOVE_ZERO_DATA_POINTS;

  return initialization_result;
}

void
MzIntegrationParams::initialize(double min_mz,
                                double max_mz,
                                MzIntegrationParams::BinningType binning_type,
                                pappso::PrecisionPtr bin_size_model,
                                int bin_size_divisor,
                                int decimal_places,
                                bool remove_zero_val_data_points,
                                QObject *parent)
{
  setSmallestMz(min_mz);
  setGreatestMz(max_mz);
  setBinningType(binning_type);
  setBinSizeModel(bin_size_model);
  setBinSizeDivisor(bin_size_divisor);
  setDecimalPlaces(decimal_places);
  setRemoveZeroValDataPoints(remove_zero_val_data_points);

  setParent(parent);
}

void
MzIntegrationParams::initialize(const MzIntegrationParams &other,
                                QObject *parent)
{
  setSmallestMz(other.m_smallestMz);
  setGreatestMz(other.m_greatestMz);
  setBinningType(other.m_binningType);
  setBinSizeModel(other.m_binSizeModel);
  setBinSizeDivisor(other.m_binSizeDivisor);
  setDecimalPlaces(other.m_decimalPlaces);
  setRemoveZeroValDataPoints(other.m_removeZeroValDataPoints);

  setParent(parent);
}

// For use in JavaScript.
void
MzIntegrationParams::initialize(const MzIntegrationParams *other_p,
                                QObject *parent)
{
  Q_ASSERT(other_p != nullptr);
  initialize(*other_p, parent);
}

// Initialize this MzIntegrationParams using other but only for members that
// were effectively set upon reading from a text string (see initialize
// (QString)).
void
MzIntegrationParams::initialize(MzIntegrationParams &other,
                                InitializationResult initialization_results)
{
  if(static_cast<bool>(
       initialization_results &
       pappso::MzIntegrationParams::InitializationResult::BINNING_TYPE))
    setBinningType(other.m_binningType);

  if(static_cast<bool>(
       initialization_results &
       pappso::MzIntegrationParams::InitializationResult::BIN_SIZE_MODEL))
    setBinSizeModel(other.m_binSizeModel);

  if(static_cast<bool>(
       initialization_results &
       pappso::MzIntegrationParams::InitializationResult::BIN_SIZE_DIVISOR))
    setBinSizeDivisor(other.m_binSizeDivisor);

  if(static_cast<bool>(
       initialization_results &
       pappso::MzIntegrationParams::InitializationResult::DECIMAL_PLACES))
    setDecimalPlaces(other.m_decimalPlaces);
}

void
MzIntegrationParams::setSmallestMz(double value)
{
  if(m_smallestMz != value)
    m_smallestMz = value;

  emit smallestMzChanged();
}

void
MzIntegrationParams::updateSmallestMz(double value)
{
  if(value == m_smallestMz)
    return;

  m_smallestMz = value;
  emit smallestMzChanged();
}

double
MzIntegrationParams::getSmallestMz() const
{
  return m_smallestMz;
}

void
MzIntegrationParams::setGreatestMz(double value)
{
  if(m_greatestMz == value)
    return;

  m_greatestMz = value;
  emit greatestMzChanged();
}

void
MzIntegrationParams::updateGreatestMz(double value)
{
  if(value > m_greatestMz)
    {
      m_greatestMz = value;
      emit greatestMzChanged();
    }
}

double
MzIntegrationParams::getGreatestMz() const
{
  return m_greatestMz;
}

// This m/z value is used to create bins phased with another previously
// performed integration. Note that the lowerAnchor m/z value might be
// smaller than the m_smallestMz value. See the bins creation function.
void
MzIntegrationParams::setLowerAnchorMz(double value)
{
  if(m_lowerAnchorMz == value)
    return;

  m_lowerAnchorMz = value;
  emit lowerAnchorMzChanged();
}

void
MzIntegrationParams::updateLowerAnchorMz(double value)
{
  if(value == m_lowerAnchorMz)
    return;

  m_lowerAnchorMz = value;
  emit lowerAnchorMzChanged();
}

double
MzIntegrationParams::getLowerAnchorMz() const
{
  return m_lowerAnchorMz;
}

// This m/z value is used to create bins phased with another previously
// performed integration. Note that the upperAnchor m/z value might be
// greater than the m_greatestMz value. See the bins creation function.
void
MzIntegrationParams::setUpperAnchorMz(double value)
{
  if(m_upperAnchorMz == value)
    return;

  m_upperAnchorMz = value;
  emit upperAnchorMzChanged();
}

void
MzIntegrationParams::updateUpperAnchorMz(double value)
{
  if(value == m_upperAnchorMz)
    return;

  m_upperAnchorMz = value;
  emit upperAnchorMzChanged();
}

double
MzIntegrationParams::getUpperAnchorMz() const
{
  return m_upperAnchorMz;
}

void
MzIntegrationParams::setMzValues(double smallest, double greatest)
{
  setSmallestMz(smallest);
  setGreatestMz(greatest);
}

void
MzIntegrationParams::setBinningType(
  MzIntegrationParams::BinningType binning_type)
{
  if(m_binningType == binning_type)
    return;

  m_binningType = binning_type;
  emit binningTypeChanged();
}

MzIntegrationParams::BinningType
MzIntegrationParams::getBinningType() const
{
  return m_binningType;
}

int
MzIntegrationParams::getDecimalPlaces() const
{
  return m_decimalPlaces;
}

void
MzIntegrationParams::setBinSizeModel(pappso::PrecisionPtr bin_size_model_p)
{
  if(m_binSizeModel == bin_size_model_p)
    return;

  m_binSizeModel = bin_size_model_p;

  if(m_binSizeModel == nullptr)
    m_binSizeModel = pappso::PrecisionFactory::getResInstance(40000);

  emit binSizeModelChanged();
}

pappso::PrecisionPtr
MzIntegrationParams::getBinSizeModel() const
{
  return m_binSizeModel;
}

void
MzIntegrationParams::setBinSizeDivisor(int divisor)
{
  if(m_binSizeDivisor == divisor)
    return;

  m_binSizeDivisor = divisor;
  emit binSizeDivisorChanged();
}

int
MzIntegrationParams::getBinSizeDivisor() const
{
  return m_binSizeDivisor;
}

void
MzIntegrationParams::setDecimalPlaces(int decimal_places)
{
  if(m_decimalPlaces == decimal_places)
    return;

  m_decimalPlaces = decimal_places;
  emit decimalPlacesChanged();
}

void
MzIntegrationParams::setRemoveZeroValDataPoints(bool removeOrNot)
{
  if(m_removeZeroValDataPoints == removeOrNot)
    return;

  m_removeZeroValDataPoints = removeOrNot;
  emit removeZeroValDataPointsChanged();
}

bool
MzIntegrationParams::isRemoveZeroValDataPoints() const
{
  return m_removeZeroValDataPoints;
}

//! Reset the instance to default values.
void
MzIntegrationParams::reset()
{
  // Each function handles the emission of the corresponding changed
  // notification.
  setSmallestMz(std::numeric_limits<double>::max());
  setGreatestMz(std::numeric_limits<double>::min());
  setBinningType(BinningType::ARBITRARY);
  setBinSizeModel(pappso::PrecisionFactory::getResInstance(40000));
  setBinSizeDivisor(1);
  setDecimalPlaces(-1);
  setRemoveZeroValDataPoints(true);
}

bool
MzIntegrationParams::isValid() const
{
  int errors = 0;

  if(m_binningType != MzIntegrationParams::BinningType::NONE)
    {
      // qDebug() << "m_smallestMz:" << m_smallestMz;
      // qDebug() << "smallest is max:" << (m_smallestMz ==
      // std::numeric_limits<double>::max());
      errors += (m_smallestMz == std::numeric_limits<double>::max() ? 1 : 0);

      // qDebug() << "m_greatestMz:" << m_greatestMz;
      // qDebug() << "greatest is min:" << (m_greatestMz ==
      // std::numeric_limits<double>::min());
      errors += (m_greatestMz == std::numeric_limits<double>::min() ? 1 : 0);
    }

  if(errors)
    {
      qCritical() << "The m/z integration parameters are invalid.";
    }

  return !errors;
}

bool
MzIntegrationParams::hasValidMzRange() const
{
  return (m_smallestMz < std::numeric_limits<double>::max()) &&
         (m_greatestMz >= std::numeric_limits<double>::min());
}

std::vector<double>
MzIntegrationParams::createBins()
{
  // qDebug() << "mp_precision:" << mp_precision->toString();

  std::vector<double> bins;

  if(m_binningType == MzIntegrationParams::BinningType::NONE)
    {
      // If no binning is to be performed, fine.
      return bins;
    }
  else if(m_binningType == MzIntegrationParams::BinningType::ARBITRARY)
    {
      // Use only data in the MzIntegrationParams member data.
      return createArbitraryBins();
    }
  else if(m_binningType == MzIntegrationParams::BinningType::DATA_BASED)
    {
      qFatal() << "Programming error. "
                  "Please use the createBins(pappso::MassSpectrumCstSPtr "
                  "mass_spectrum_csp) overload.";
    }

  return bins;
}

std::vector<double>
MzIntegrationParams::createBins(pappso::MassSpectrumCstSPtr mass_spectrum_csp)
{
  // qDebug();

  std::vector<double> bins;

  if(m_binningType == MzIntegrationParams::BinningType::NONE)
    {
      // If no binning is to be performed, fine.
      return bins;
    }
  else if(m_binningType == MzIntegrationParams::BinningType::ARBITRARY)
    {
      // Use only data in the MzIntegrationParams member data.
      return createArbitraryBins();
    }
  else if(m_binningType == MzIntegrationParams::BinningType::DATA_BASED)
    {
      // qDebug();

      // Use the first spectrum to perform the data-based bins

      return createDataBasedBins(mass_spectrum_csp);
    }

  return bins;
}

std::vector<double>
MzIntegrationParams::createArbitraryBins()
{
  // Now starts the tricky stuff. Depending on the binning size model, we need
  // to take diverse actions.

  if(!isValid())
    qFatal() << "Programming error. The MzIntegrationParams::BinningLogic is "
                "not valid, cannot "
                "create bins.";

  // qDebug() << "Binning logic:" << toString();

  double min_mz = m_smallestMz;
  double max_mz = m_greatestMz;

  // qDebug() << " Target min_mz:" << min_mz << "and max_mz:" << max_mz;

  double bin_size;

  // We will need to account for the bin size divisor, that is set to 6 as the
  // default because that is the empirical observation that it gives the most
  // nicely shaped peaks.
  Q_ASSERT(m_binSizeDivisor > 0);

  bin_size = m_binSizeModel->delta(min_mz) / m_binSizeDivisor;

  // qDebug() << QString::asprintf(
  //"binSize is the precision delta for min_mz: %.6f\n", binSize);

  // Only compute the decimal places if they were not configured already.
  if(m_decimalPlaces == -1)
    {
      // qDebug() << "Now checking how many decimal places are needed.";

      // We want as many decimal places as there are 0s between the integral
      // part of the double and the first non-0 cipher.  For example, if
      // binSize is 0.004, zero decimals is 2 and m_decimalPlaces is set to 3,
      // because we want decimals up to 4 included.

      m_decimalPlaces = pappso::Utils::zeroDecimalsInValue(bin_size) + 1;

      // qDebug() << "With binSize" << bin_size
      //          << " m_decimalPlaces was computed to be:" << m_decimalPlaces;
    }
  // else
  // qDebug() << "m_decimalPlaces: " << m_decimalPlaces;

  // Now that we have defined the value of m_decimalPlaces, let's use that
  // value.

  double first_mz = ceil((min_mz * std::pow(10, m_decimalPlaces)) - 0.49) /
                    pow(10, m_decimalPlaces);
  double last_mz =
    ceil((max_mz * pow(10, m_decimalPlaces)) - 0.49) / pow(10, m_decimalPlaces);

  // qDebug() << qSetRealNumberPrecision(6)
  //          << "After having accounted for the decimals, new min/max values:"
  //          << "Very first data point:" << first_mz << "Very last data point
  //          to reach:" << last_mz;

  // Instantiate the vector of mz double_s that we'll feed with the bins.

  std::vector<double> bins;
  double previous_mz_bin;
  double current_mz;

  // There is a gotcha here: if the user has set the m_lowerAnchorMz value,
  // then that means that we want to start phasing the bins to that value.
  if(m_lowerAnchorMz != std::numeric_limits<double>::max())
    {
      // qDebug() << qSetRealNumberPrecision(6) << "a lower anchor was set at"
      //          << m_lowerAnchorMz;

      double lower_anchor_mz =
        ceil((m_lowerAnchorMz * std::pow(10, m_decimalPlaces)) - 0.49) /
        pow(10, m_decimalPlaces);

      // qDebug() << qSetRealNumberPrecision(6)
      //          << "after decimals accounting, becomes:" << lower_anchor_mz;

      // There are two situations:

      if(lower_anchor_mz <= first_mz)
        {
          // qDebug()
          //   << "lower anchor m/z is smaller than min_mz, min_mz becomes
          //   lower_anchor_mz;";

          // 1. If m_lowerAnchorMz is smaller than m_smallestMz, then we need to
          // use the former value as the start value for the bin calculations.
          // We do not compute specific lower anchor-related bins, first_mz just
          // becomes lower_anchor_mz.

          first_mz = lower_anchor_mz;
        }
      else
        {
          // lower_anchor_mz > first_mz

          // qDebug() << "Special case where we need to prepend bins.";

          // 2. If m_lowerAnchorMz is greater than the m_smallestMz, then we
          // need to compute in reverse the bins so that the bins are phased
          // with that m_anchorMz value. Note that at then end of the
          // computation of these lower anchor-related bins, the normal course
          // of operation will start at first_mz, which is still
          // lower_anchor_mz.

          QList<double> prepended_bins;

          previous_mz_bin = lower_anchor_mz;

          while(previous_mz_bin > first_mz)
            {
              current_mz =
                previous_mz_bin -
                (m_binSizeModel->delta(previous_mz_bin) / m_binSizeDivisor);

              // Now apply on the obtained mz value the decimals that were
              // either set or computed earlier.

              double current_rounded_mz =
                ceil((current_mz * pow(10, m_decimalPlaces)) - 0.49) /
                pow(10, m_decimalPlaces);

              // qDebug() << QString::asprintf(
              //"current_mz: %.6f and current_rounded_mz: %.6f and
              // previous_mzBin "
              //": % .6f\n ",
              // current_mz,
              // current_rounded_mz,
              // previous_mz_bin);

              // If rounding makes the new value identical to the previous one,
              // then that means that we need to decrease roughness.

              if(current_rounded_mz == previous_mz_bin)
                {
                  ++m_decimalPlaces;

                  current_rounded_mz =
                    ceil((current_mz * pow(10, m_decimalPlaces)) - 0.49) /
                    pow(10, m_decimalPlaces);

                  // qDebug().noquote()
                  //<< "Had to increment decimal places by one while creating
                  // the bins " "in MzIntegrationParams::BinningType::ARBITRARY
                  // mode..";
                }

              // qDebug() << qSetRealNumberPrecision(6) << "Prepending bin"
              //          << current_rounded_mz;

              prepended_bins.prepend(current_rounded_mz);

              // Use the local_mz value for the storage of the previous mz bin.
              previous_mz_bin = current_rounded_mz;
            }
          // End of
          // while(previous_mz_bin > anchor_mz)

          // Now set the bins to the vector that we'll use later.
          bins.assign(prepended_bins.begin(), prepended_bins.end());

          // #if 0

          QString bins_with_delta = binsToStringWithDeltas(bins);

          QString file_name =
            "/tmp/prependedMassSpecBins.txt-at-" +
            QDateTime::currentDateTime().toString("yyyyMMdd-HH-mm-ss");

          // qDebug() << "Writing the list of prepended bins setup in the "
          //             "mass spectrum in file "
          //          << file_name;

          Q_ASSERT(Utils::writeToFile(bins_with_delta, file_name));

          // #endif
        }
    }
  // End of
  // if(m_lowerAnchorMz != std::numeric_limits<double>::max())

  // At this point, either we have already set some bins because
  // anchor_mz was greater than min_mz, or we start fresh.

  // We have the same gotcha as for the lower anchor m/z value, this
  // time with the upper anchor m/z value:

  if(m_upperAnchorMz != std::numeric_limits<double>::max())
    {
      // qDebug() << qSetRealNumberPrecision(6) << "an upper anchor was set at"
      //          << m_upperAnchorMz;

      double upper_anchor_mz =
        ceil((m_upperAnchorMz * std::pow(10, m_decimalPlaces)) - 0.49) /
        pow(10, m_decimalPlaces);

      // qDebug() << qSetRealNumberPrecision(6)
      //          << "after decimals accounting, becomes:" << upper_anchor_mz;

      if(upper_anchor_mz > last_mz)
        {
          // qDebug() << "since upper anchor m/z is greater than last_mz,
          // last_mz becomes "
          //             "upper_anchor_mz.";

          last_mz = upper_anchor_mz;
        }
    }

  // Store that very first value for later use in the loop.
  // The bins are nothing more than:
  //
  // 1. The first mz (that is the smallest mz value found in all the spectra
  // 2. A sequence of mz values corresponding to that first mz value
  // incremented by the bin size.

  // Seed the root of the bin vector with the first mz value rounded above as
  // requested. that first mz value is actually anchor_mz.
  previous_mz_bin = first_mz;

  bins.push_back(previous_mz_bin);

  // Now continue adding mz values until we have reached the end of the
  // spectrum, that is the max_mz value, as converted using the decimals to
  // last_mz.

  // debugCount value used below for debugging purposes.
  // int debugCount = 0;

  while(previous_mz_bin <= last_mz)
    {
      // qDebug() << "Now starting the bin creation loop.";

      // Calculate dynamically the precision delta according to the current mz
      // value.

      // double precision_delta = mp_precision->delta(previous_mz_bin);
      // qDebug() << "precision_delta: " << precision_delta;

      // In certain circumstances, the bin size is not enough to properly render
      // hyper-high resolution data (like the theoretical isotopic cluster data
      // generated in silico). In that case, the bin size, computed using the
      // precision object, is divided by the m_binSizeDivisor, which normally is
      // set to 6 as the default because that is the empirical observation that
      // it gives the most nicely shaped peaks.

      current_mz = previous_mz_bin +
                   (m_binSizeModel->delta(previous_mz_bin) / m_binSizeDivisor);

      // qDebug() << QString::asprintf(
      //"previous_mzBin: %.6f and current_mz: %.6f\n",
      // previous_mz_bin,
      // current_mz);

      // Now apply on the obtained mz value the decimals that were either set
      // or computed earlier.

      double current_rounded_mz =
        ceil((current_mz * pow(10, m_decimalPlaces)) - 0.49) /
        pow(10, m_decimalPlaces);

      // qDebug() << QString::asprintf(
      //"current_mz: %.6f and current_rounded_mz: %.6f and previous_mzBin "
      //": % .6f\n ",
      // current_mz,
      // current_rounded_mz,
      // previous_mz_bin);

      // If rounding makes the new value identical to the previous one, then
      // that means that we need to decrease roughness.

      if(current_rounded_mz == previous_mz_bin)
        {
          ++m_decimalPlaces;

          current_rounded_mz =
            ceil((current_mz * pow(10, m_decimalPlaces)) - 0.49) /
            pow(10, m_decimalPlaces);

          // qDebug().noquote()
          //<< "Had to increment decimal places by one while creating the bins "
          //"in MzIntegrationParams::BinningType::ARBITRARY mode..";
        }

      bins.push_back(current_rounded_mz);

      // Use the local_mz value for the storage of the previous mz bin.
      previous_mz_bin = current_rounded_mz;
    }

#if 0

  QString bins_with_delta = binsToStringWithDelta(bins);

  QString file_name = "/tmp/massSpecArbitraryBins.txt-at-" +
                      QDateTime::currentDateTime().toString("yyyyMMdd-HH-mm-ss");

  qDebug() << "Writing the list of bins setup in the mass spectrum in file " << file_name;

  Q_ASSERT(Utils::writeToFile(bins_with_delta, file_name));

#endif

  // qDebug() << "Prepared " << bins.size() << "arbitrary bins starting with mz"
  // << bins.front()
  //          << "ending with mz" << bins.back();

  return bins;
}

std::vector<double>
MzIntegrationParams::createDataBasedBins(
  pappso::MassSpectrumCstSPtr mass_spectrum_csp)
{
  // The mass spectrum passed as parameters has intrinsic bins in it. These bins
  // are nothing else than the succession of m/z values in it.

  // The very first thing to do is replicate that spectrum into a local copy
  // of bins (simple double values), and during that step recompute the m/z
  // value (x value) according to the decimals settings in this instance, if
  // that is required (that is m_decimalPlaces != -1).

  QList<double> bins;
  QList<double> deltas;
  QList<double> resolutions;

  double left_mz_value  = 0;
  double right_mz_value = 0;
  double mz_delta       = 0;
  double resolution     = 0;

  // We need three bins to compute a number of values.
  if(mass_spectrum_csp->size() < 3)
    {
      std::vector<double> bins_vector(bins.constBegin(), bins.constEnd());
      return bins_vector;
    }

  // Make sure the spectrum is sorted, as this function takes for granted
  // that the DataPoint instances are sorted in ascending x (== mz) value
  // order.
  pappso::MassSpectrum mass_spectrum_copy = *mass_spectrum_csp;
  mass_spectrum_copy.sortMz();

  std::vector<pappso::DataPoint>::const_iterator iterator_const =
    mass_spectrum_copy.cbegin();

  // We need to see the first value
  left_mz_value = iterator_const->x;
  if(m_decimalPlaces != -1)
    left_mz_value = ceil((left_mz_value * pow(10, m_decimalPlaces)) - 0.49) /
                    pow(10, m_decimalPlaces);

  qDebug() << qSetRealNumberPrecision(6)
           << "left_mz_value in the template mass spectrum:" << left_mz_value;

  bins.append(left_mz_value);
  // No delta nor resolution can be computed for the first m/z
  // because their computation requires to compare the second m/z with
  // the first m/z, and in turn for all the remaining m/z values.
  deltas.append(0.0);
  resolutions.append(0.0);

  // We used the first bin, go to the next!
  ++iterator_const;
  while(iterator_const != mass_spectrum_copy.cend())
    {
      right_mz_value = iterator_const->x;

      qDebug() << qSetRealNumberPrecision(6)
               << "right_mz_value:" << right_mz_value;

      if(m_decimalPlaces != -1)
        right_mz_value =
          ceil((right_mz_value * pow(10, m_decimalPlaces)) - 0.49) /
          pow(10, m_decimalPlaces);

      bins.append(right_mz_value);

      mz_delta = right_mz_value - left_mz_value;
      deltas.append(mz_delta);
      Q_ASSERT(mz_delta != 0.0);

      resolution = right_mz_value / mz_delta;
      resolutions.append(resolution);

      left_mz_value = right_mz_value;
      ++iterator_const;
    }

  // At this point we have the bins that match those in the template mass
  // spectrum.

  // #if 0

  // We want to check them.
  std::vector<double> bins_vector(bins.constBegin(), bins.constEnd());

  QString bins_with_delta = binsToStringWithDeltas(bins_vector);

  QString file_name =
    "/tmp/massSpecDataBasedTemplateBinsWithDeltas.txt-at-" +
    QDateTime::currentDateTime().toString("yyyyMMdd-HH-mm-ss");

  // qDebug() << "Writing the list of bins setup in the mass spectrum in file "
  // << file_name;

  Q_ASSERT(Utils::writeToFile(bins_with_delta, file_name));

  // #endif

  // Done, we now have bins and deltas in between each bin and its following
  // bin.

  // Now, there are specificities:
  //
  // If the user has set m_lowerAnchorMz and/or m_upperAnchorMz, then we
  // still have to work to do. That is because the user wants us to keep
  // the binning as found in the template mass spectrum passed as parameter, but
  // they also want that the m/z range of the bins be potentially larger, if
  // either m_lowerAnchorMz is smaller than the first bin value or if
  // m_upperAnchorMz is larger than the last bin value, or BOTH.

  // The difficulty here is that we need to know the logic that has been used
  // in the first place for the generation of the bins in the template mass
  // spectrum.

  // Some metrics:

  // We could check if the bins have the same width all along the m/z range.

  // Delta should be constant if the bins were calculated with dalton logic.

  bool uniform_deltas      = false;
  std::size_t deltas_count = deltas.size();
  // We cannot really take the first delta because it is 0.00000.
  double first_delta       = deltas.at(1);
  double middle_delta      = deltas.at(deltas_count / 2);
  double last_delta        = deltas.at(deltas_count - 1);

  if(first_delta == middle_delta && middle_delta == last_delta)
    uniform_deltas = true;

  // qDebug() << "Deltas are uniform or not?" << uniform_deltas;
  // qDebug() << qSetRealNumberPrecision(6) << "First delta:" << first_delta
  //          << "Middle delta:" << middle_delta << "Last delta:" << last_delta;

  // Resolution should be constant if the bins were calculated with resolution
  // logic. Remember : Res = m/z / Delta m/z.
  bool uniform_resolutions      = false;
  std::size_t resolutions_count = resolutions.size();
  // We cannot really take the first resolution because it is 0.00000.
  double first_resolution       = resolutions.at(01);
  double middle_resolution      = resolutions.at(resolutions_count / 2);
  double last_resolution        = resolutions.at(resolutions_count - 1);

  if(first_resolution == middle_resolution &&
     middle_resolution == last_resolution)
    uniform_resolutions = true;

  // qDebug() << "Deltas are uniform or not?" << uniform_resolutions;
  // qDebug() << qSetRealNumberPrecision(6) << "First resolution:" <<
  // first_resolution
  //          << "Middle resolution:" << middle_resolution << "Last resolution:"
  //          << last_resolution;

  // Now that we have the metric about the bins, we can start checking the other
  // parameters of the creation of bins:
  // There are two pairs of values in this instance:
  // m_smallestMz vs m_greatestMz
  // and
  // m_lowerAnchorMz vs m_upperAnchorMz

  // Which one of these pairs should we account for, or both ?

  // In theory, m_smallestMz and m_greatestMz were designed for the arbitrary
  // binning method, but it is imaginable that one would want to combine to and
  // existing spectrum a new spectrum that is larger than the existing
  // one, either on the left or on the right.

  // Then, there is the other pair: m_lowerAnchorMz vs m_upperAnchorMz. In the
  // end what could be done is to take the smallest of all the values for the
  // left end of the integration bins and the greatest for the right end of the
  // integration bins.

  double first_bin_mz = bins.first();
  double last_bin_mz  = bins.last();

  double smallest_mz = std::min(m_smallestMz, m_lowerAnchorMz);
  double greatest_mz = std::max(m_greatestMz, m_upperAnchorMz);

  // qDebug() << qSetRealNumberPrecision(6) << "smallest_mz:" << smallest_mz
  //          << "greatest_mz:" << greatest_mz;

  // Now we can check if the [smallest_mz -- greatest_mz] m/z value range
  // extends farther on the left or on the right or both compared to the bins
  // range.

  // Now that we have the real smallest and greatest m/z values that might be
  // std::max() and std::min() respectively if not set by the user. So all we
  // have to do is check them against the first and last bin values.

  // The list of prepended bins on the left of the existing bins. By starting
  // with first_bin_mz, we maintain the phase with the bins in the mass spectrum
  // passed as parameter.

  if(smallest_mz < first_bin_mz)
    {
      // qDebug() << "A starting m/z value is smaller than the first bin in the
      // "
      //             "spectrum";

      // We will have to craft bins to the left of first_bin_mz up to
      // smallest_mz. There are going to be different strategies depending on
      // what we discovered above. We will prepend the newly created bins to the
      // bins QList (thanks QList).

      double bin_mz = first_bin_mz;

      if(uniform_deltas)
        {
          // That is the simplest situation.

          while(bin_mz > smallest_mz)
            {
              double new_bin_mz = bin_mz - first_delta;
              bins.prepend(new_bin_mz);
              bin_mz = new_bin_mz;
            }
        }
      else if(uniform_resolutions)
        {
          // Remember: Res = m/z / Delta m/z.

          while(bin_mz > smallest_mz)
            {
              double new_bin_mz = bin_mz - (bin_mz / first_resolution);
              bins.prepend(new_bin_mz);
              bin_mz = new_bin_mz;
            }
        }
    }

  // Now make sure we append any necessary bin (same logic as above).
  // By starting with last_bin_mz, we maintain the phase with the bins in the
  // mass spectrum passed as parameter.

  if(greatest_mz > last_bin_mz)
    {
      // qDebug() << "A stopping m/z value is greater than the last bin in the "
      //             "spectrum";

      // Craft bins to the right of the bins m/z range, that is append
      // new bins.

      double bin_mz = last_bin_mz;

      if(uniform_deltas)
        {
          // That is the simplest situation.

          while(bin_mz < greatest_mz)
            {
              double new_bin_mz = bin_mz + first_delta;
              bins.append(new_bin_mz);
              bin_mz = new_bin_mz;
            }
        }
      else if(uniform_resolutions)
        {
          // Remember: Res = m/z / Delta m/z.

          while(bin_mz > smallest_mz)
            {
              double new_bin_mz = bin_mz + (bin_mz / first_resolution);
              bins.append(new_bin_mz);
              bin_mz = new_bin_mz;
            }
        }
    }

  // At this point we should have bins over the full range of m/z values
  // required by the user and in phase with those transmitted via the
  // mass spectrum argument.

  // Convert to std::vector<double> and return.
  std::vector<double> full_bins_vector(bins.constBegin(), bins.constEnd());

#if 0

  bins_with_delta = binsToStringWithDeltas(full_bins_vector);

  file_name = "/tmp/massSpecDataBasedFullBinsWithDeltas.txt-at-" +
              QDateTime::currentDateTime().toString("yyyyMMdd-HH-mm-ss");

  qDebug() << "Writing the list of bins setup in the mass spectrum in file "
           << file_name;

  Q_ASSERT(Utils::writeToFile(bins_with_delta, file_name));

#endif

  // qDebug() << "Prepared " << bins.size() << "data-based bins starting with
  // mz"
  //          << bins.front() << "ending with mz" << bins.back();

  return full_bins_vector;
}

std::vector<double>
MzIntegrationParams::createDataBasedBinsOld(
  pappso::MassSpectrumCstSPtr mass_spectrum_csp)
{
  // The bins must be calculated using those in mass_spectrum_csp as a template.

  // qDebug();

  std::vector<double> bins;

  if(mass_spectrum_csp->size() < 2)
    return bins;

  // Make sure the spectrum is sorted, as this function takes for granted
  // that the DataPoint instances are sorted in ascending x (== mz) value
  // order.
  pappso::MassSpectrum sorted_mass_spectrum = *mass_spectrum_csp;
  sorted_mass_spectrum.sortMz();

  double min_mz = m_smallestMz;

  // qDebug() << "The min_mz:" << min_mz;

  if(m_decimalPlaces != -1)
    min_mz = ceil((min_mz * pow(10, m_decimalPlaces)) - 0.49) /
             pow(10, m_decimalPlaces);

  // Two values for the definition of a MassSpectrumBin.

  // The first value of the mz range that defines the bin. This value is part
  // of the bin.
  double start_mz_in = min_mz;

  // The second value of the mz range that defines the bin. This value is
  // *not* part of the bin.
  double end_mz_out;

  std::vector<pappso::DataPoint>::const_iterator it =
    sorted_mass_spectrum.begin();

  double prev_mz = it->x;

  if(m_decimalPlaces != -1)
    prev_mz = ceil((prev_mz * pow(10, m_decimalPlaces)) - 0.49) /
              pow(10, m_decimalPlaces);

  ++it;

  while(it != sorted_mass_spectrum.end())
    {
      double next_mz = it->x;

      if(m_decimalPlaces != -1)
        next_mz = ceil((next_mz * pow(10, m_decimalPlaces)) - 0.49) /
                  pow(10, m_decimalPlaces);

      double step = next_mz - prev_mz;
      end_mz_out  = start_mz_in + step;

      if(m_decimalPlaces != -1)
        end_mz_out = ceil((end_mz_out * pow(10, m_decimalPlaces)) - 0.49) /
                     pow(10, m_decimalPlaces);

      // The data point that is crafted has a 0 y-value. The binning must
      // indeed not create artificial intensity data.

      // qDebug() << "Pushing back bin:" << start_mz_in << end_mz_out;

      bins.push_back(start_mz_in);

      // Prepare next bin
      start_mz_in = end_mz_out;

      // Update prev_mz to be the current one for next iteration.
      prev_mz = next_mz;

      // Now go to the next DataPoint instance.
      ++it;
    }

#if 0

  QString fileName = "/tmp/massSpecDataBasedBins.txt";

  qDebug() << "Writing the list of bins setup in the "
  "mass spectrum in file "
  << fileName;

  QFile file(fileName);
  file.open(QIODevice::WriteOnly);

  QTextStream fileStream(&file);

  for(auto &&bin : m_bins)
    fileStream << QString("[%1-%2]\n")
    .arg(bin.startMzIn, 0, 'f', 10)
    .arg(bin.endMzOut, 0, 'f', 10);

  fileStream.flush();
  file.close();

  qDebug() << "elements."
  << "starting with mz" << m_bins.front().startMzIn << "ending with mz"
  << m_bins.back().endMzOut;

#endif

  return bins;
}

// This is for documentation, not for outputting the string used in the
// settings that are saved on disk.
QString
MzIntegrationParams::toString(int offset, const QString &spacer) const
{
  // The space-containing string that reflects the offset at which
  // new text lines should be added to start with.
  QString offset_lead;

  for(int iter = 0; iter < offset; ++iter)
    offset_lead += spacer;

  QString text  = offset_lead;
  text         += "m/z integration parameters:\n";

  QString new_lead = QString("%1%2").arg(offset_lead, spacer);

  text += new_lead;
  if(m_smallestMz != std::numeric_limits<double>::max())
    text.append(
      QString::asprintf("Smallest (first) m/z: %.6f\n", m_smallestMz));

  text += new_lead;
  if(m_greatestMz != std::numeric_limits<double>::min())
    text.append(QString::asprintf("Greatest (last) m/z: %.6f\n", m_greatestMz));

  text += new_lead;
  if(m_lowerAnchorMz != std::numeric_limits<double>::max())
    text.append(QString::asprintf("Lower anchor m/z: %.6f\n", m_lowerAnchorMz));

  text += new_lead;
  if(m_upperAnchorMz != std::numeric_limits<double>::max())
    text.append(QString::asprintf("Upper anchor m/z: %.6f\n", m_upperAnchorMz));

  text += new_lead;
  text += QString("Remove 0-val data points: %1\n")
            .arg(m_removeZeroValDataPoints ? "true" : "false");

  text += new_lead;
  text.append("Binning logic:\n");

  new_lead += spacer;

  text += new_lead;
  text.append(
    QString("Binning type:%1\n").arg(::pappso::binningTypeMap[m_binningType]));

  text += new_lead;
  text.append(QString("Bin size model: %1\n").arg(m_binSizeModel->toString()));

  text += new_lead;
  text.append(QString("Bin size divisor: %2\n").arg(m_binSizeDivisor));

  text += new_lead;
  text.append(QString("Decimal places: %1\n").arg(m_decimalPlaces));

  return text;
}

// This version is used to craft the string that enables the initialization.
QString
MzIntegrationParams::toString() const
{
  QString text;

  // In the string for saving settings, we do not ouput the m/z values.
  text.append(
    QString("Binning type:%1\n").arg(::pappso::binningTypeMap[m_binningType]));
  text.append(QString("Bin size model: %1\n").arg(m_binSizeModel->toString()));
  text.append(QString("Bin size divisor: %2\n").arg(m_binSizeDivisor));
  text.append(QString("Decimal places:%1\n").arg(m_decimalPlaces));
  text.append(
    QString("Remove 0-val data points:%1\n").arg(m_removeZeroValDataPoints));

  // qDebug().noquote() << "Returning text:\n" << text;

  return text;
}

QString
MzIntegrationParams::binsToStringWithDeltas(
  const std::vector<double> bins) const
{
  QString bins_with_delta;
  double previous_bin_value = 0;

  for(auto &&value : bins)
    {
      double delta     = value - previous_bin_value;
      bins_with_delta += QString("%1 - %2\n")
                           .arg(value, 0, 'f', m_decimalPlaces)
                           .arg(delta, 0, 'f', m_decimalPlaces);
      previous_bin_value = value;
    }

  return bins_with_delta;
}

void
MzIntegrationParams::registerJsConstructor(QJSEngine *engine)
{
  // qDebug() << "registerJsConstructor for MzIntegrationParams to QJSEngine.";

  if(engine == nullptr)
    {
      qFatal() << "Cannot register class: engine is null";
    }

  QJSValue pappso_root_property;
  QJSValue pappso_enums_property;

  if(engine->globalObject().hasProperty("pappso"))
    {
      qDebug() << "Global object property 'pappso' already exists.";
      pappso_root_property = engine->globalObject().property("pappso");

      if(pappso_root_property.hasProperty("Enums"))
        {
          pappso_enums_property = pappso_root_property.property("Enums");
        }
      else
        {
          pappso_enums_property = engine->newObject();
          pappso_root_property.setProperty("Enums", pappso_enums_property);
        }
    }
  else
    {
      qDebug() << "Global object property 'pappso' not found.";
      pappso_root_property  = engine->newObject();
      pappso_enums_property = engine->newObject();
      pappso_root_property.setProperty("Enums", pappso_enums_property);
      engine->globalObject().setProperty("pappso", pappso_root_property);
    }

  QJSValue enumObject;

  // First the BinningType enum
  enumObject = engine->newObject();

  enumObject.setProperty(
    "NONE", static_cast<int>(pappso::MzIntegrationParams::BinningType::NONE));
  enumObject.setProperty(
    "DATA_BASED",
    static_cast<int>(pappso::MzIntegrationParams::BinningType::DATA_BASED));
  enumObject.setProperty(
    "ARBITRARY",
    static_cast<int>(pappso::MzIntegrationParams::BinningType::ARBITRARY));

  pappso_enums_property.setProperty("BinningType", enumObject);

  // Second the InitializationResult enum
  enumObject = engine->newObject();
  enumObject.setProperty(
    "DEFAULT",
    static_cast<int>(
      pappso::MzIntegrationParams::InitializationResult::DEFAULT));
  enumObject.setProperty(
    "fromSettingsBinSizeModelPartial",
    static_cast<int>(pappso::MzIntegrationParams::InitializationResult::
                       BINNING_LOGIC_PARTIAL));
  enumObject.setProperty(
    "fromSettingsBinSizeModelFull",
    static_cast<int>(
      pappso::MzIntegrationParams::InitializationResult::BINNING_LOGIC_FULL));
  enumObject.setProperty(
    "fromSettingsFull",
    static_cast<int>(pappso::MzIntegrationParams::InitializationResult::FULL));

  pappso_enums_property.setProperty("InitializationResult", enumObject);

  // Finally gegister the class meta object as a constructor for the class
  QJSValue jsMetaObject =
    engine->newQMetaObject(&MzIntegrationParams::staticMetaObject);
  engine->globalObject().setProperty("MzIntegrationParams", jsMetaObject);
}

} // namespace pappso
