// ****************************************************************************
//
//          Aevol - An in silico experimental evolution platform
//
// ****************************************************************************
//
// Copyright: See the AUTHORS file provided with the package or <www.aevol.fr>
// Web: http://www.aevol.fr/
// E-mail: See <http://www.aevol.fr/contact/>
// Original Authors : Guillaume Beslon, Carole Knibbe, David Parsons
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ****************************************************************************

#ifndef AEVOL_CHECKPOINTHANDLER_H_
#define AEVOL_CHECKPOINTHANDLER_H_

#include <filesystem>
#include <fstream>
#include <iostream>
#include <optional>
#include <string>

#include "AeTime.h"
#include "ae_types.h"
#include "CheckpointData.h"
#include "ExpSetup.h"
#include "Grid.h"
#include "Individual.h"
#include "indiv_idxs_map.h"
#include "io/fasta/FastaReader.h"
#include "io/fasta/FastaWriter.h"
#include "mxifstream.h"
#include "mxofstream.h"
#include "phenotype/PhenotypicTarget.h"

namespace aevol {

class CheckpointHandler {
 protected:
  CheckpointHandler();
  CheckpointHandler(const CheckpointHandler&)            = delete;
  CheckpointHandler(CheckpointHandler&&)                 = delete;
  CheckpointHandler& operator=(const CheckpointHandler&) = delete;
  CheckpointHandler& operator=(CheckpointHandler&&)      = delete;
  CheckpointHandler(const std::filesystem::path& chkpts_dir);
 public:
  virtual ~CheckpointHandler()                           = default;

  static auto make_default() -> std::unique_ptr<CheckpointHandler>;
  static auto make_from_ckpts_dir(const std::filesystem::path& chkpts_dir) -> std::unique_ptr<CheckpointHandler>;
  static auto make_from_ckpt_file(const std::filesystem::path& chkpt_file) -> std::unique_ptr<CheckpointHandler>;

  static auto read_checkpoint(const std::filesystem::path& chkp_path,
                              const std::filesystem::path& pop_path,
                              const std::filesystem::path& setup_path) -> CheckpointData;
  static auto read_checkpoint(const std::filesystem::path& chkp_path) -> CheckpointData;
  auto read_checkpoint(aevol::time_type time) const -> CheckpointData;
  void write_checkpoint(aevol::time_type time,
                        const indivs_idxs_sorted_map_t& indivs_idxs_map,
                        const Grid& grid,
                        const PhenotypicTarget& target,
                        const ExpSetup& exp_setup,
                        time_type checkpoint_frequency,
                        std::optional<time_type> tree_output_frequency,
                        std::optional<time_type> best_indiv_stats_output_frequency,
                        std::optional<time_type> whole_pop_stats_output_frequency);

 protected:
  std::filesystem::path chkpts_dir_;

  auto make_chkp_dir_path(time_type t) const -> std::filesystem::path;
  auto make_chkp_files_paths(time_type t) const -> std::array<std::filesystem::path, 3>;
  static auto make_chkp_files_paths(const std::filesystem::path& chkpt_file) -> std::array<std::filesystem::path, 3>;

  static void check_compatibility(std::string header, std::filesystem::path path);

  static auto read_population(FastaReader& fasta_reader, std::size_t population_size) -> fasta_structured_population_t;
  void write_indivs_idxs_map(FastaWriter& fasta_writer, const indivs_idxs_sorted_map_t& indivs_idxs_map) const;
};

}  // namespace aevol

#endif  // AEVOL_CHECKPOINTHANDLER_H_
