/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Christopher Moore <moore@free.fr>
 */

#include <stddef.h>
#include <stdint.h>
#define SIMDE_TEST_X86_AVX512_INSN insert
#include "../../test.h"
#include "../test-avx.h"
#include "test-avx512.h"
#include <simde/hedley.h>
#include <simde/simde-constify.h>
#include <simde/simde-common.h>
#include <simde/x86/sse.h>
#include <simde/x86/sse2.h>
#include <simde/x86/avx.h>
#include <simde/x86/avx512/types.h>
#include <simde/x86/avx512/loadu.h>
#include <simde/x86/avx512/setzero.h>
#include <simde/x86/avx512/insert.h>

static int
test_simde_mm256_insertf32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const float a[8];
    const float b[4];
    const int imm8;
    const float r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -670.18), SIMDE_FLOAT32_C(  -587.76), SIMDE_FLOAT32_C(   -58.80), SIMDE_FLOAT32_C(    62.59),
        SIMDE_FLOAT32_C(   821.40), SIMDE_FLOAT32_C(   417.06), SIMDE_FLOAT32_C(  -273.85), SIMDE_FLOAT32_C(   162.94) },
      { SIMDE_FLOAT32_C(   241.96), SIMDE_FLOAT32_C(   -98.97), SIMDE_FLOAT32_C(   312.15), SIMDE_FLOAT32_C(  -536.28) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(   241.96), SIMDE_FLOAT32_C(   -98.97), SIMDE_FLOAT32_C(   312.15), SIMDE_FLOAT32_C(  -536.28),
        SIMDE_FLOAT32_C(   821.40), SIMDE_FLOAT32_C(   417.06), SIMDE_FLOAT32_C(  -273.85), SIMDE_FLOAT32_C(   162.94) } },
    { { SIMDE_FLOAT32_C(   -98.20), SIMDE_FLOAT32_C(   691.09), SIMDE_FLOAT32_C(   641.04), SIMDE_FLOAT32_C(   547.12),
        SIMDE_FLOAT32_C(  -977.88), SIMDE_FLOAT32_C(  -563.28), SIMDE_FLOAT32_C(   521.13), SIMDE_FLOAT32_C(   849.91) },
      { SIMDE_FLOAT32_C(  -426.02), SIMDE_FLOAT32_C(  -190.76), SIMDE_FLOAT32_C(  -345.63), SIMDE_FLOAT32_C(   910.49) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   -98.20), SIMDE_FLOAT32_C(   691.09), SIMDE_FLOAT32_C(   641.04), SIMDE_FLOAT32_C(   547.12),
        SIMDE_FLOAT32_C(  -426.02), SIMDE_FLOAT32_C(  -190.76), SIMDE_FLOAT32_C(  -345.63), SIMDE_FLOAT32_C(   910.49) } },
    { { SIMDE_FLOAT32_C(  -288.03), SIMDE_FLOAT32_C(    60.78), SIMDE_FLOAT32_C(   444.53), SIMDE_FLOAT32_C(   915.99),
        SIMDE_FLOAT32_C(  -271.51), SIMDE_FLOAT32_C(   774.34), SIMDE_FLOAT32_C(  -671.77), SIMDE_FLOAT32_C(   669.69) },
      { SIMDE_FLOAT32_C(  -163.07), SIMDE_FLOAT32_C(  -850.36), SIMDE_FLOAT32_C(    86.75), SIMDE_FLOAT32_C(   563.08) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(  -163.07), SIMDE_FLOAT32_C(  -850.36), SIMDE_FLOAT32_C(    86.75), SIMDE_FLOAT32_C(   563.08),
        SIMDE_FLOAT32_C(  -271.51), SIMDE_FLOAT32_C(   774.34), SIMDE_FLOAT32_C(  -671.77), SIMDE_FLOAT32_C(   669.69) } },
    { { SIMDE_FLOAT32_C(  -671.29), SIMDE_FLOAT32_C(  -535.89), SIMDE_FLOAT32_C(  -375.27), SIMDE_FLOAT32_C(  -207.57),
        SIMDE_FLOAT32_C(   232.76), SIMDE_FLOAT32_C(   526.53), SIMDE_FLOAT32_C(  -516.48), SIMDE_FLOAT32_C(  -126.20) },
      { SIMDE_FLOAT32_C(    73.65), SIMDE_FLOAT32_C(  -494.36), SIMDE_FLOAT32_C(   310.52), SIMDE_FLOAT32_C(  -405.22) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(    73.65), SIMDE_FLOAT32_C(  -494.36), SIMDE_FLOAT32_C(   310.52), SIMDE_FLOAT32_C(  -405.22),
        SIMDE_FLOAT32_C(   232.76), SIMDE_FLOAT32_C(   526.53), SIMDE_FLOAT32_C(  -516.48), SIMDE_FLOAT32_C(  -126.20) } },
    { { SIMDE_FLOAT32_C(   884.50), SIMDE_FLOAT32_C(   404.02), SIMDE_FLOAT32_C(     9.93), SIMDE_FLOAT32_C(   794.98),
        SIMDE_FLOAT32_C(   601.50), SIMDE_FLOAT32_C(   721.90), SIMDE_FLOAT32_C(  -144.23), SIMDE_FLOAT32_C(    46.03) },
      { SIMDE_FLOAT32_C(   637.89), SIMDE_FLOAT32_C(   584.26), SIMDE_FLOAT32_C(  -179.63), SIMDE_FLOAT32_C(   966.12) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   884.50), SIMDE_FLOAT32_C(   404.02), SIMDE_FLOAT32_C(     9.93), SIMDE_FLOAT32_C(   794.98),
        SIMDE_FLOAT32_C(   637.89), SIMDE_FLOAT32_C(   584.26), SIMDE_FLOAT32_C(  -179.63), SIMDE_FLOAT32_C(   966.12) } },
    { { SIMDE_FLOAT32_C(   657.30), SIMDE_FLOAT32_C(  -884.24), SIMDE_FLOAT32_C(  -659.30), SIMDE_FLOAT32_C(   220.38),
        SIMDE_FLOAT32_C(   428.34), SIMDE_FLOAT32_C(  -330.59), SIMDE_FLOAT32_C(   684.49), SIMDE_FLOAT32_C(  -946.93) },
      { SIMDE_FLOAT32_C(   461.84), SIMDE_FLOAT32_C(   -82.76), SIMDE_FLOAT32_C(   579.60), SIMDE_FLOAT32_C(   945.37) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   657.30), SIMDE_FLOAT32_C(  -884.24), SIMDE_FLOAT32_C(  -659.30), SIMDE_FLOAT32_C(   220.38),
        SIMDE_FLOAT32_C(   461.84), SIMDE_FLOAT32_C(   -82.76), SIMDE_FLOAT32_C(   579.60), SIMDE_FLOAT32_C(   945.37) } },
    { { SIMDE_FLOAT32_C(  -346.75), SIMDE_FLOAT32_C(  -548.99), SIMDE_FLOAT32_C(   101.56), SIMDE_FLOAT32_C(   248.03),
        SIMDE_FLOAT32_C(  -193.44), SIMDE_FLOAT32_C(   -13.94), SIMDE_FLOAT32_C(  -347.95), SIMDE_FLOAT32_C(   816.49) },
      { SIMDE_FLOAT32_C(  -218.96), SIMDE_FLOAT32_C(  -746.45), SIMDE_FLOAT32_C(   538.39), SIMDE_FLOAT32_C(   636.81) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(  -218.96), SIMDE_FLOAT32_C(  -746.45), SIMDE_FLOAT32_C(   538.39), SIMDE_FLOAT32_C(   636.81),
        SIMDE_FLOAT32_C(  -193.44), SIMDE_FLOAT32_C(   -13.94), SIMDE_FLOAT32_C(  -347.95), SIMDE_FLOAT32_C(   816.49) } },
    { { SIMDE_FLOAT32_C(   176.28), SIMDE_FLOAT32_C(   221.06), SIMDE_FLOAT32_C(  -880.06), SIMDE_FLOAT32_C(   142.40),
        SIMDE_FLOAT32_C(  -524.99), SIMDE_FLOAT32_C(   777.24), SIMDE_FLOAT32_C(   258.16), SIMDE_FLOAT32_C(  -184.29) },
      { SIMDE_FLOAT32_C(    -2.38), SIMDE_FLOAT32_C(  -313.50), SIMDE_FLOAT32_C(   485.12), SIMDE_FLOAT32_C(  -317.89) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   176.28), SIMDE_FLOAT32_C(   221.06), SIMDE_FLOAT32_C(  -880.06), SIMDE_FLOAT32_C(   142.40),
        SIMDE_FLOAT32_C(    -2.38), SIMDE_FLOAT32_C(  -313.50), SIMDE_FLOAT32_C(   485.12), SIMDE_FLOAT32_C(  -317.89) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 a = simde_mm256_loadu_ps(test_vec[i].a);
    simde__m128 b = simde_mm_loadu_ps(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_f32x8(simde_mm256_insertf32x4(a, b, 0), simde_mm256_loadu_ps(test_vec[i].r), 1); break;
      case 1: simde_test_x86_assert_equal_f32x8(simde_mm256_insertf32x4(a, b, 1), simde_mm256_loadu_ps(test_vec[i].r), 1); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256 a = simde_test_x86_random_f32x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m128 b = simde_test_x86_random_f32x4(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m256 r;
    SIMDE_CONSTIFY_2_(simde_mm256_insertf32x4, r, (HEDLEY_UNREACHABLE(), simde_mm256_setzero_ps ()), imm8, a, b);

    simde_test_x86_write_f32x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_insertf64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const double a[4];
    const double b[2];
    const int imm8;
    const double r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(    92.73), SIMDE_FLOAT64_C(   740.04), SIMDE_FLOAT64_C(   533.23), SIMDE_FLOAT64_C(   294.13) },
      { SIMDE_FLOAT64_C(  -708.99), SIMDE_FLOAT64_C(    -0.12) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(    92.73), SIMDE_FLOAT64_C(   740.04), SIMDE_FLOAT64_C(  -708.99), SIMDE_FLOAT64_C(    -0.12) } },
    { { SIMDE_FLOAT64_C(  -835.54), SIMDE_FLOAT64_C(   749.37), SIMDE_FLOAT64_C(  -462.73), SIMDE_FLOAT64_C(  -191.02) },
      { SIMDE_FLOAT64_C(   863.00), SIMDE_FLOAT64_C(  -607.22) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -835.54), SIMDE_FLOAT64_C(   749.37), SIMDE_FLOAT64_C(   863.00), SIMDE_FLOAT64_C(  -607.22) } },
    { { SIMDE_FLOAT64_C(  -432.89), SIMDE_FLOAT64_C(  -302.10), SIMDE_FLOAT64_C(   846.45), SIMDE_FLOAT64_C(  -294.89) },
      { SIMDE_FLOAT64_C(  -379.45), SIMDE_FLOAT64_C(  -299.98) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(  -379.45), SIMDE_FLOAT64_C(  -299.98), SIMDE_FLOAT64_C(   846.45), SIMDE_FLOAT64_C(  -294.89) } },
    { { SIMDE_FLOAT64_C(   694.63), SIMDE_FLOAT64_C(   401.75), SIMDE_FLOAT64_C(   466.59), SIMDE_FLOAT64_C(  -902.68) },
      { SIMDE_FLOAT64_C(  -466.37), SIMDE_FLOAT64_C(   152.24) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(  -466.37), SIMDE_FLOAT64_C(   152.24), SIMDE_FLOAT64_C(   466.59), SIMDE_FLOAT64_C(  -902.68) } },
    { { SIMDE_FLOAT64_C(    47.03), SIMDE_FLOAT64_C(  -746.00), SIMDE_FLOAT64_C(   468.62), SIMDE_FLOAT64_C(  -860.24) },
      { SIMDE_FLOAT64_C(   994.04), SIMDE_FLOAT64_C(     1.85) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(    47.03), SIMDE_FLOAT64_C(  -746.00), SIMDE_FLOAT64_C(   994.04), SIMDE_FLOAT64_C(     1.85) } },
    { { SIMDE_FLOAT64_C(  -714.94), SIMDE_FLOAT64_C(  -998.26), SIMDE_FLOAT64_C(  -610.33), SIMDE_FLOAT64_C(  -550.48) },
      { SIMDE_FLOAT64_C(   751.11), SIMDE_FLOAT64_C(   -73.06) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   751.11), SIMDE_FLOAT64_C(   -73.06), SIMDE_FLOAT64_C(  -610.33), SIMDE_FLOAT64_C(  -550.48) } },
    { { SIMDE_FLOAT64_C(   614.11), SIMDE_FLOAT64_C(   319.71), SIMDE_FLOAT64_C(   712.82), SIMDE_FLOAT64_C(  -818.79) },
      { SIMDE_FLOAT64_C(  -982.39), SIMDE_FLOAT64_C(   559.27) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   614.11), SIMDE_FLOAT64_C(   319.71), SIMDE_FLOAT64_C(  -982.39), SIMDE_FLOAT64_C(   559.27) } },
    { { SIMDE_FLOAT64_C(  -361.84), SIMDE_FLOAT64_C(  -740.71), SIMDE_FLOAT64_C(  -339.70), SIMDE_FLOAT64_C(  -667.21) },
      { SIMDE_FLOAT64_C(   661.04), SIMDE_FLOAT64_C(  -873.11) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -361.84), SIMDE_FLOAT64_C(  -740.71), SIMDE_FLOAT64_C(   661.04), SIMDE_FLOAT64_C(  -873.11) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256d a = simde_mm256_loadu_pd(test_vec[i].a);
    simde__m128d b = simde_mm_loadu_pd(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_f64x4(simde_mm256_insertf64x2(a, b, 0), simde_mm256_loadu_pd(test_vec[i].r), 1); break;
      case 1: simde_test_x86_assert_equal_f64x4(simde_mm256_insertf64x2(a, b, 1), simde_mm256_loadu_pd(test_vec[i].r), 1); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256d a = simde_test_x86_random_f64x4(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m128d b = simde_test_x86_random_f64x2(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m256d r;
    SIMDE_CONSTIFY_2_(simde_mm256_insertf64x2, r, (HEDLEY_UNREACHABLE(), simde_mm256_setzero_pd ()), imm8, a, b);

    simde_test_x86_write_f64x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_inserti32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[8];
    const int32_t b[4];
    const int imm8;
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(   893376555), -INT32_C(  1573482179), -INT32_C(   129172268), -INT32_C(  1203788666),  INT32_C(   333078369), -INT32_C(  1338370037), -INT32_C(  1220003357), -INT32_C(  2039482533) },
      {  INT32_C(   230469583),  INT32_C(   347075136),  INT32_C(  1947007982),  INT32_C(     2902943) },
       INT32_C(           0),
      {  INT32_C(   230469583),  INT32_C(   347075136),  INT32_C(  1947007982),  INT32_C(     2902943),  INT32_C(   333078369), -INT32_C(  1338370037), -INT32_C(  1220003357), -INT32_C(  2039482533) } },
    { {  INT32_C(   515183622), -INT32_C(  2029951410), -INT32_C(  1511868242), -INT32_C(   680171224),  INT32_C(   387416613),  INT32_C(   738536241), -INT32_C(  2100528585), -INT32_C(  1389507418) },
      {  INT32_C(   785113824), -INT32_C(   155857592), -INT32_C(  1382246267),  INT32_C(   646189313) },
       INT32_C(           1),
      {  INT32_C(   515183622), -INT32_C(  2029951410), -INT32_C(  1511868242), -INT32_C(   680171224),  INT32_C(   785113824), -INT32_C(   155857592), -INT32_C(  1382246267),  INT32_C(   646189313) } },
    { { -INT32_C(   960217701), -INT32_C(  1124208574), -INT32_C(  1989967684), -INT32_C(  1888939859), -INT32_C(  1462200101), -INT32_C(   449917363),  INT32_C(  2078727018), -INT32_C(    82965408) },
      { -INT32_C(  1933389239),  INT32_C(  2135474115), -INT32_C(   334976193), -INT32_C(  1753451845) },
       INT32_C(           0),
      { -INT32_C(  1933389239),  INT32_C(  2135474115), -INT32_C(   334976193), -INT32_C(  1753451845), -INT32_C(  1462200101), -INT32_C(   449917363),  INT32_C(  2078727018), -INT32_C(    82965408) } },
    { {  INT32_C(   576143188),  INT32_C(  1233993069),  INT32_C(   816384035), -INT32_C(   377904105),  INT32_C(   648807782), -INT32_C(   110810034), -INT32_C(  1498066380),  INT32_C(   581979342) },
      { -INT32_C(   112981877),  INT32_C(  1749209413), -INT32_C(   241636390), -INT32_C(   153480561) },
       INT32_C(           1),
      {  INT32_C(   576143188),  INT32_C(  1233993069),  INT32_C(   816384035), -INT32_C(   377904105), -INT32_C(   112981877),  INT32_C(  1749209413), -INT32_C(   241636390), -INT32_C(   153480561) } },
    { { -INT32_C(  1285219193), -INT32_C(   739811455),  INT32_C(  1621200404),  INT32_C(  1189856062), -INT32_C(   645143544),  INT32_C(   297006118), -INT32_C(  1633573748),  INT32_C(   112629631) },
      {  INT32_C(   884546483), -INT32_C(  1912037254),  INT32_C(  1844357422),  INT32_C(  1974721133) },
       INT32_C(           0),
      {  INT32_C(   884546483), -INT32_C(  1912037254),  INT32_C(  1844357422),  INT32_C(  1974721133), -INT32_C(   645143544),  INT32_C(   297006118), -INT32_C(  1633573748),  INT32_C(   112629631) } },
    { {  INT32_C(   870665791), -INT32_C(  1497369086),  INT32_C(   774200983),  INT32_C(   803285780), -INT32_C(  2052450588), -INT32_C(   944556002),  INT32_C(    20193319),  INT32_C(   331327956) },
      { -INT32_C(    96033544),  INT32_C(   849347995),  INT32_C(  2002896483), -INT32_C(   693681423) },
       INT32_C(           0),
      { -INT32_C(    96033544),  INT32_C(   849347995),  INT32_C(  2002896483), -INT32_C(   693681423), -INT32_C(  2052450588), -INT32_C(   944556002),  INT32_C(    20193319),  INT32_C(   331327956) } },
    { { -INT32_C(  1988732079),  INT32_C(   800079374),  INT32_C(   470004082),  INT32_C(   353638001),  INT32_C(  1638927964),  INT32_C(  1959125934), -INT32_C(  2040120252),  INT32_C(   887045347) },
      { -INT32_C(  1514252905),  INT32_C(   114585235), -INT32_C(  1859987680),  INT32_C(  1235629805) },
       INT32_C(           0),
      { -INT32_C(  1514252905),  INT32_C(   114585235), -INT32_C(  1859987680),  INT32_C(  1235629805),  INT32_C(  1638927964),  INT32_C(  1959125934), -INT32_C(  2040120252),  INT32_C(   887045347) } },
    { {  INT32_C(   988982103), -INT32_C(  1401002128),  INT32_C(   160433357),  INT32_C(   966837475), -INT32_C(   238271102), -INT32_C(   250490342),  INT32_C(   719299316), -INT32_C(  1620170680) },
      {  INT32_C(  1138319827), -INT32_C(  1779411000),  INT32_C(  1067352156), -INT32_C(   948421052) },
       INT32_C(           1),
      {  INT32_C(   988982103), -INT32_C(  1401002128),  INT32_C(   160433357),  INT32_C(   966837475),  INT32_C(  1138319827), -INT32_C(  1779411000),  INT32_C(  1067352156), -INT32_C(   948421052) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_mm256_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi32(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_i32x8(simde_mm256_inserti32x4(a, b, 0), simde_mm256_loadu_epi32(test_vec[i].r)); break;
      case 1: simde_test_x86_assert_equal_i32x8(simde_mm256_inserti32x4(a, b, 1), simde_mm256_loadu_epi32(test_vec[i].r)); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m128i b = simde_test_x86_random_i32x4();
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m256i r;
    SIMDE_CONSTIFY_2_(simde_mm256_inserti32x4, r, (HEDLEY_UNREACHABLE(), simde_mm256_setzero_si256 ()), imm8, a, b);

    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_inserti64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[4];
    const int64_t b[2];
    const int imm8;
    const int64_t r[4];
  } test_vec[] = {
    { { -INT64_C( 7120811991911042745), -INT64_C( 4094575125621112883), -INT64_C( 2474049518318203824),  INT64_C(  449474579989682230) },
      { -INT64_C( 6871989648238114613),  INT64_C( 5273383082796504264) },
       INT32_C(           1),
      { -INT64_C( 7120811991911042745), -INT64_C( 4094575125621112883), -INT64_C( 6871989648238114613),  INT64_C( 5273383082796504264) } },
    { { -INT64_C( 8955534116004003560), -INT64_C( 4680003390623548825), -INT64_C( 8758037057842834972),  INT64_C( 2469601018279681238) },
      { -INT64_C( 1515079183755840182),  INT64_C( 7587221238685907926) },
       INT32_C(           1),
      { -INT64_C( 8955534116004003560), -INT64_C( 4680003390623548825), -INT64_C( 1515079183755840182),  INT64_C( 7587221238685907926) } },
    { {  INT64_C( 7559156236696753886), -INT64_C( 3956248630944166140), -INT64_C( 4437675630690333384),  INT64_C( 2370774954464966241) },
      {  INT64_C( 7762543739766170566), -INT64_C(  485172972706255705) },
       INT32_C(           0),
      {  INT64_C( 7762543739766170566), -INT64_C(  485172972706255705), -INT64_C( 4437675630690333384),  INT64_C( 2370774954464966241) } },
    { { -INT64_C( 1358072817506742896), -INT64_C( 8366756099261480339),  INT64_C( 1963673820706583936), -INT64_C( 8014203911510705001) },
      { -INT64_C( 4591844973139841951),  INT64_C( 5004274092193183039) },
       INT32_C(           1),
      { -INT64_C( 1358072817506742896), -INT64_C( 8366756099261480339), -INT64_C( 4591844973139841951),  INT64_C( 5004274092193183039) } },
    { { -INT64_C( 2640236244610261403),  INT64_C( 3968071544140940507),  INT64_C( 2701678943014644162),  INT64_C( 4458053285515519272) },
      { -INT64_C( 7131348020892679666),  INT64_C( 1288695040139125664) },
       INT32_C(           1),
      { -INT64_C( 2640236244610261403),  INT64_C( 3968071544140940507), -INT64_C( 7131348020892679666),  INT64_C( 1288695040139125664) } },
    { {  INT64_C( 3737877527884353644),  INT64_C( 7548957276947335045), -INT64_C( 2646207860296716215),  INT64_C( 4627515500252623123) },
      { -INT64_C( 3490335571329962355),  INT64_C( 3146687801769368222) },
       INT32_C(           1),
      {  INT64_C( 3737877527884353644),  INT64_C( 7548957276947335045), -INT64_C( 3490335571329962355),  INT64_C( 3146687801769368222) } },
    { {  INT64_C(  626012539942064341),  INT64_C(  576210447831408950), -INT64_C( 8896823771033150700),  INT64_C( 5558727784352107685) },
      { -INT64_C( 7673965129709824927), -INT64_C( 8351072871731733318) },
       INT32_C(           0),
      { -INT64_C( 7673965129709824927), -INT64_C( 8351072871731733318), -INT64_C( 8896823771033150700),  INT64_C( 5558727784352107685) } },
    { { -INT64_C( 3804467464192100500), -INT64_C( 7482572155800838090),  INT64_C( 8992998100330586233),  INT64_C( 7565345871000625556) },
      {  INT64_C( 5747482740982696440),  INT64_C( 1163686370054899273) },
       INT32_C(           1),
      { -INT64_C( 3804467464192100500), -INT64_C( 7482572155800838090),  INT64_C( 5747482740982696440),  INT64_C( 1163686370054899273) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_mm256_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi64(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_i64x4(simde_mm256_inserti64x2(a, b, 0), simde_mm256_loadu_epi64(test_vec[i].r)); break;
      case 1: simde_test_x86_assert_equal_i64x4(simde_mm256_inserti64x2(a, b, 1), simde_mm256_loadu_epi64(test_vec[i].r)); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m128i b = simde_test_x86_random_i64x2();
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m256i r;
    SIMDE_CONSTIFY_2_(simde_mm256_inserti64x2, r, (HEDLEY_UNREACHABLE(), simde_mm256_setzero_si256 ()), imm8, a, b);

    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_insertf32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const float a[16];
    const float b[4];
    const int imm8;
    const float r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(    49.76), SIMDE_FLOAT32_C(   389.26), SIMDE_FLOAT32_C(  -452.29), SIMDE_FLOAT32_C(   340.39),
        SIMDE_FLOAT32_C(   211.54), SIMDE_FLOAT32_C(   889.58), SIMDE_FLOAT32_C(  -900.10), SIMDE_FLOAT32_C(    98.13),
        SIMDE_FLOAT32_C(   447.35), SIMDE_FLOAT32_C(   -51.48), SIMDE_FLOAT32_C(   470.89), SIMDE_FLOAT32_C(   910.51),
        SIMDE_FLOAT32_C(   131.62), SIMDE_FLOAT32_C(  -964.59), SIMDE_FLOAT32_C(   682.36), SIMDE_FLOAT32_C(   749.02) },
      { SIMDE_FLOAT32_C(   340.16), SIMDE_FLOAT32_C(  -813.43), SIMDE_FLOAT32_C(  -771.31), SIMDE_FLOAT32_C(  -692.35) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(    49.76), SIMDE_FLOAT32_C(   389.26), SIMDE_FLOAT32_C(  -452.29), SIMDE_FLOAT32_C(   340.39),
        SIMDE_FLOAT32_C(   340.16), SIMDE_FLOAT32_C(  -813.43), SIMDE_FLOAT32_C(  -771.31), SIMDE_FLOAT32_C(  -692.35),
        SIMDE_FLOAT32_C(   447.35), SIMDE_FLOAT32_C(   -51.48), SIMDE_FLOAT32_C(   470.89), SIMDE_FLOAT32_C(   910.51),
        SIMDE_FLOAT32_C(   131.62), SIMDE_FLOAT32_C(  -964.59), SIMDE_FLOAT32_C(   682.36), SIMDE_FLOAT32_C(   749.02) } },
    { { SIMDE_FLOAT32_C(  -427.36), SIMDE_FLOAT32_C(   732.74), SIMDE_FLOAT32_C(   625.72), SIMDE_FLOAT32_C(   739.95),
        SIMDE_FLOAT32_C(   819.83), SIMDE_FLOAT32_C(  -320.08), SIMDE_FLOAT32_C(   977.23), SIMDE_FLOAT32_C(  -372.96),
        SIMDE_FLOAT32_C(  -274.49), SIMDE_FLOAT32_C(   868.97), SIMDE_FLOAT32_C(   676.80), SIMDE_FLOAT32_C(  -885.23),
        SIMDE_FLOAT32_C(  -583.32), SIMDE_FLOAT32_C(    17.19), SIMDE_FLOAT32_C(   326.31), SIMDE_FLOAT32_C(  -693.73) },
      { SIMDE_FLOAT32_C(   117.08), SIMDE_FLOAT32_C(  -575.57), SIMDE_FLOAT32_C(   753.61), SIMDE_FLOAT32_C(  -934.39) },
       INT32_C(           2),
      { SIMDE_FLOAT32_C(  -427.36), SIMDE_FLOAT32_C(   732.74), SIMDE_FLOAT32_C(   625.72), SIMDE_FLOAT32_C(   739.95),
        SIMDE_FLOAT32_C(   819.83), SIMDE_FLOAT32_C(  -320.08), SIMDE_FLOAT32_C(   977.23), SIMDE_FLOAT32_C(  -372.96),
        SIMDE_FLOAT32_C(   117.08), SIMDE_FLOAT32_C(  -575.57), SIMDE_FLOAT32_C(   753.61), SIMDE_FLOAT32_C(  -934.39),
        SIMDE_FLOAT32_C(  -583.32), SIMDE_FLOAT32_C(    17.19), SIMDE_FLOAT32_C(   326.31), SIMDE_FLOAT32_C(  -693.73) } },
    { { SIMDE_FLOAT32_C(   664.13), SIMDE_FLOAT32_C(   197.23), SIMDE_FLOAT32_C(   930.74), SIMDE_FLOAT32_C(   346.49),
        SIMDE_FLOAT32_C(   -53.75), SIMDE_FLOAT32_C(   270.90), SIMDE_FLOAT32_C(   533.05), SIMDE_FLOAT32_C(   174.93),
        SIMDE_FLOAT32_C(   578.55), SIMDE_FLOAT32_C(  -280.00), SIMDE_FLOAT32_C(   747.58), SIMDE_FLOAT32_C(   311.29),
        SIMDE_FLOAT32_C(  -654.28), SIMDE_FLOAT32_C(   487.52), SIMDE_FLOAT32_C(   131.12), SIMDE_FLOAT32_C(    25.64) },
      { SIMDE_FLOAT32_C(   464.76), SIMDE_FLOAT32_C(   758.16), SIMDE_FLOAT32_C(   751.15), SIMDE_FLOAT32_C(   333.73) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(   664.13), SIMDE_FLOAT32_C(   197.23), SIMDE_FLOAT32_C(   930.74), SIMDE_FLOAT32_C(   346.49),
        SIMDE_FLOAT32_C(   -53.75), SIMDE_FLOAT32_C(   270.90), SIMDE_FLOAT32_C(   533.05), SIMDE_FLOAT32_C(   174.93),
        SIMDE_FLOAT32_C(   578.55), SIMDE_FLOAT32_C(  -280.00), SIMDE_FLOAT32_C(   747.58), SIMDE_FLOAT32_C(   311.29),
        SIMDE_FLOAT32_C(   464.76), SIMDE_FLOAT32_C(   758.16), SIMDE_FLOAT32_C(   751.15), SIMDE_FLOAT32_C(   333.73) } },
    { { SIMDE_FLOAT32_C(   865.92), SIMDE_FLOAT32_C(   750.41), SIMDE_FLOAT32_C(  -547.86), SIMDE_FLOAT32_C(   192.23),
        SIMDE_FLOAT32_C(  -943.32), SIMDE_FLOAT32_C(   569.23), SIMDE_FLOAT32_C(   616.66), SIMDE_FLOAT32_C(   810.29),
        SIMDE_FLOAT32_C(   634.83), SIMDE_FLOAT32_C(   511.98), SIMDE_FLOAT32_C(   474.42), SIMDE_FLOAT32_C(  -167.94),
        SIMDE_FLOAT32_C(   442.72), SIMDE_FLOAT32_C(  -179.10), SIMDE_FLOAT32_C(   778.31), SIMDE_FLOAT32_C(  -286.37) },
      { SIMDE_FLOAT32_C(  -646.04), SIMDE_FLOAT32_C(   -46.76), SIMDE_FLOAT32_C(  -707.82), SIMDE_FLOAT32_C(    73.96) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(   865.92), SIMDE_FLOAT32_C(   750.41), SIMDE_FLOAT32_C(  -547.86), SIMDE_FLOAT32_C(   192.23),
        SIMDE_FLOAT32_C(  -943.32), SIMDE_FLOAT32_C(   569.23), SIMDE_FLOAT32_C(   616.66), SIMDE_FLOAT32_C(   810.29),
        SIMDE_FLOAT32_C(   634.83), SIMDE_FLOAT32_C(   511.98), SIMDE_FLOAT32_C(   474.42), SIMDE_FLOAT32_C(  -167.94),
        SIMDE_FLOAT32_C(  -646.04), SIMDE_FLOAT32_C(   -46.76), SIMDE_FLOAT32_C(  -707.82), SIMDE_FLOAT32_C(    73.96) } },
    { { SIMDE_FLOAT32_C(   603.48), SIMDE_FLOAT32_C(   419.67), SIMDE_FLOAT32_C(  -811.66), SIMDE_FLOAT32_C(  -265.40),
        SIMDE_FLOAT32_C(  -554.69), SIMDE_FLOAT32_C(   653.10), SIMDE_FLOAT32_C(  -507.24), SIMDE_FLOAT32_C(  -803.54),
        SIMDE_FLOAT32_C(   -13.17), SIMDE_FLOAT32_C(   927.72), SIMDE_FLOAT32_C(  -937.62), SIMDE_FLOAT32_C(  -262.76),
        SIMDE_FLOAT32_C(  -620.13), SIMDE_FLOAT32_C(   254.61), SIMDE_FLOAT32_C(  -206.08), SIMDE_FLOAT32_C(   949.09) },
      { SIMDE_FLOAT32_C(  -128.74), SIMDE_FLOAT32_C(  -395.80), SIMDE_FLOAT32_C(   583.93), SIMDE_FLOAT32_C(  -616.75) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(  -128.74), SIMDE_FLOAT32_C(  -395.80), SIMDE_FLOAT32_C(   583.93), SIMDE_FLOAT32_C(  -616.75),
        SIMDE_FLOAT32_C(  -554.69), SIMDE_FLOAT32_C(   653.10), SIMDE_FLOAT32_C(  -507.24), SIMDE_FLOAT32_C(  -803.54),
        SIMDE_FLOAT32_C(   -13.17), SIMDE_FLOAT32_C(   927.72), SIMDE_FLOAT32_C(  -937.62), SIMDE_FLOAT32_C(  -262.76),
        SIMDE_FLOAT32_C(  -620.13), SIMDE_FLOAT32_C(   254.61), SIMDE_FLOAT32_C(  -206.08), SIMDE_FLOAT32_C(   949.09) } },
    { { SIMDE_FLOAT32_C(  -584.01), SIMDE_FLOAT32_C(   825.97), SIMDE_FLOAT32_C(  -100.48), SIMDE_FLOAT32_C(  -805.70),
        SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   253.48), SIMDE_FLOAT32_C(   147.54), SIMDE_FLOAT32_C(  -168.22),
        SIMDE_FLOAT32_C(  -672.57), SIMDE_FLOAT32_C(   848.36), SIMDE_FLOAT32_C(  -564.74), SIMDE_FLOAT32_C(   747.11),
        SIMDE_FLOAT32_C(  -963.29), SIMDE_FLOAT32_C(   169.86), SIMDE_FLOAT32_C(  -807.58), SIMDE_FLOAT32_C(   689.81) },
      { SIMDE_FLOAT32_C(   662.62), SIMDE_FLOAT32_C(  -611.12), SIMDE_FLOAT32_C(  -323.36), SIMDE_FLOAT32_C(   590.34) },
       INT32_C(           2),
      { SIMDE_FLOAT32_C(  -584.01), SIMDE_FLOAT32_C(   825.97), SIMDE_FLOAT32_C(  -100.48), SIMDE_FLOAT32_C(  -805.70),
        SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   253.48), SIMDE_FLOAT32_C(   147.54), SIMDE_FLOAT32_C(  -168.22),
        SIMDE_FLOAT32_C(   662.62), SIMDE_FLOAT32_C(  -611.12), SIMDE_FLOAT32_C(  -323.36), SIMDE_FLOAT32_C(   590.34),
        SIMDE_FLOAT32_C(  -963.29), SIMDE_FLOAT32_C(   169.86), SIMDE_FLOAT32_C(  -807.58), SIMDE_FLOAT32_C(   689.81) } },
    { { SIMDE_FLOAT32_C(   413.88), SIMDE_FLOAT32_C(   970.21), SIMDE_FLOAT32_C(   705.87), SIMDE_FLOAT32_C(  -792.21),
        SIMDE_FLOAT32_C(   919.30), SIMDE_FLOAT32_C(  -422.86), SIMDE_FLOAT32_C(  -188.00), SIMDE_FLOAT32_C(   503.23),
        SIMDE_FLOAT32_C(   -39.62), SIMDE_FLOAT32_C(  -109.38), SIMDE_FLOAT32_C(   919.22), SIMDE_FLOAT32_C(  -213.64),
        SIMDE_FLOAT32_C(   790.14), SIMDE_FLOAT32_C(  -886.48), SIMDE_FLOAT32_C(   325.96), SIMDE_FLOAT32_C(    43.62) },
      { SIMDE_FLOAT32_C(   261.07), SIMDE_FLOAT32_C(  -842.26), SIMDE_FLOAT32_C(   371.05), SIMDE_FLOAT32_C(   109.43) },
       INT32_C(           2),
      { SIMDE_FLOAT32_C(   413.88), SIMDE_FLOAT32_C(   970.21), SIMDE_FLOAT32_C(   705.87), SIMDE_FLOAT32_C(  -792.21),
        SIMDE_FLOAT32_C(   919.30), SIMDE_FLOAT32_C(  -422.86), SIMDE_FLOAT32_C(  -188.00), SIMDE_FLOAT32_C(   503.23),
        SIMDE_FLOAT32_C(   261.07), SIMDE_FLOAT32_C(  -842.26), SIMDE_FLOAT32_C(   371.05), SIMDE_FLOAT32_C(   109.43),
        SIMDE_FLOAT32_C(   790.14), SIMDE_FLOAT32_C(  -886.48), SIMDE_FLOAT32_C(   325.96), SIMDE_FLOAT32_C(    43.62) } },
    { { SIMDE_FLOAT32_C(   118.16), SIMDE_FLOAT32_C(   146.14), SIMDE_FLOAT32_C(   762.86), SIMDE_FLOAT32_C(   310.58),
        SIMDE_FLOAT32_C(  -164.06), SIMDE_FLOAT32_C(   425.49), SIMDE_FLOAT32_C(   699.47), SIMDE_FLOAT32_C(   512.58),
        SIMDE_FLOAT32_C(    15.83), SIMDE_FLOAT32_C(  -849.27), SIMDE_FLOAT32_C(   -73.54), SIMDE_FLOAT32_C(   -13.96),
        SIMDE_FLOAT32_C(   856.60), SIMDE_FLOAT32_C(   134.26), SIMDE_FLOAT32_C(   -94.65), SIMDE_FLOAT32_C(  -566.26) },
      { SIMDE_FLOAT32_C(   946.26), SIMDE_FLOAT32_C(  -591.42), SIMDE_FLOAT32_C(   394.12), SIMDE_FLOAT32_C(  -163.12) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(   946.26), SIMDE_FLOAT32_C(  -591.42), SIMDE_FLOAT32_C(   394.12), SIMDE_FLOAT32_C(  -163.12),
        SIMDE_FLOAT32_C(  -164.06), SIMDE_FLOAT32_C(   425.49), SIMDE_FLOAT32_C(   699.47), SIMDE_FLOAT32_C(   512.58),
        SIMDE_FLOAT32_C(    15.83), SIMDE_FLOAT32_C(  -849.27), SIMDE_FLOAT32_C(   -73.54), SIMDE_FLOAT32_C(   -13.96),
        SIMDE_FLOAT32_C(   856.60), SIMDE_FLOAT32_C(   134.26), SIMDE_FLOAT32_C(   -94.65), SIMDE_FLOAT32_C(  -566.26) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m128 b = simde_mm_loadu_ps(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_f32x16(simde_mm512_insertf32x4(a, b, 0), simde_mm512_loadu_ps(test_vec[i].r), 1); break;
      case 1: simde_test_x86_assert_equal_f32x16(simde_mm512_insertf32x4(a, b, 1), simde_mm512_loadu_ps(test_vec[i].r), 1); break;
      case 2: simde_test_x86_assert_equal_f32x16(simde_mm512_insertf32x4(a, b, 2), simde_mm512_loadu_ps(test_vec[i].r), 1); break;
      case 3: simde_test_x86_assert_equal_f32x16(simde_mm512_insertf32x4(a, b, 3), simde_mm512_loadu_ps(test_vec[i].r), 1); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m128 b = simde_test_x86_random_f32x4(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 3;
    simde__m512 r;
    SIMDE_CONSTIFY_4_(simde_mm512_insertf32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), imm8, a, b);

    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_insertf32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const float src[16];
    const simde__mmask16 k;
    const float a[16];
    const float b[4];
    const int imm8;
    const float r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -628.81), SIMDE_FLOAT32_C(  -565.24), SIMDE_FLOAT32_C(   594.19), SIMDE_FLOAT32_C(  -560.82),
        SIMDE_FLOAT32_C(  -113.33), SIMDE_FLOAT32_C(    79.02), SIMDE_FLOAT32_C(  -407.15), SIMDE_FLOAT32_C(    -5.14),
        SIMDE_FLOAT32_C(  -524.08), SIMDE_FLOAT32_C(   739.17), SIMDE_FLOAT32_C(  -297.50), SIMDE_FLOAT32_C(  -223.58),
        SIMDE_FLOAT32_C(  -624.99), SIMDE_FLOAT32_C(  -790.62), SIMDE_FLOAT32_C(  -814.44), SIMDE_FLOAT32_C(  -491.40) },
      UINT16_C(61483),
      { SIMDE_FLOAT32_C(   880.52), SIMDE_FLOAT32_C(   715.39), SIMDE_FLOAT32_C(   323.47), SIMDE_FLOAT32_C(  -950.83),
        SIMDE_FLOAT32_C(   927.91), SIMDE_FLOAT32_C(   200.82), SIMDE_FLOAT32_C(  -815.97), SIMDE_FLOAT32_C(  -949.38),
        SIMDE_FLOAT32_C(   366.98), SIMDE_FLOAT32_C(  -676.42), SIMDE_FLOAT32_C(  -500.32), SIMDE_FLOAT32_C(  -538.60),
        SIMDE_FLOAT32_C(  -884.02), SIMDE_FLOAT32_C(  -129.13), SIMDE_FLOAT32_C(  -103.85), SIMDE_FLOAT32_C(   710.17) },
      { SIMDE_FLOAT32_C(   310.06), SIMDE_FLOAT32_C(   782.83), SIMDE_FLOAT32_C(  -210.81), SIMDE_FLOAT32_C(   902.91) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   880.52), SIMDE_FLOAT32_C(   715.39), SIMDE_FLOAT32_C(   594.19), SIMDE_FLOAT32_C(  -950.83),
        SIMDE_FLOAT32_C(  -113.33), SIMDE_FLOAT32_C(   782.83), SIMDE_FLOAT32_C(  -407.15), SIMDE_FLOAT32_C(    -5.14),
        SIMDE_FLOAT32_C(  -524.08), SIMDE_FLOAT32_C(   739.17), SIMDE_FLOAT32_C(  -297.50), SIMDE_FLOAT32_C(  -223.58),
        SIMDE_FLOAT32_C(  -884.02), SIMDE_FLOAT32_C(  -129.13), SIMDE_FLOAT32_C(  -103.85), SIMDE_FLOAT32_C(   710.17) } },
    { { SIMDE_FLOAT32_C(   265.11), SIMDE_FLOAT32_C(   642.08), SIMDE_FLOAT32_C(   480.18), SIMDE_FLOAT32_C(  -958.47),
        SIMDE_FLOAT32_C(  -982.91), SIMDE_FLOAT32_C(   689.56), SIMDE_FLOAT32_C(  -772.91), SIMDE_FLOAT32_C(  -474.31),
        SIMDE_FLOAT32_C(  -538.84), SIMDE_FLOAT32_C(  -330.38), SIMDE_FLOAT32_C(  -593.79), SIMDE_FLOAT32_C(  -823.45),
        SIMDE_FLOAT32_C(   993.09), SIMDE_FLOAT32_C(  -544.62), SIMDE_FLOAT32_C(  -895.54), SIMDE_FLOAT32_C(   193.91) },
      UINT16_C(57190),
      { SIMDE_FLOAT32_C(  -439.11), SIMDE_FLOAT32_C(   -37.01), SIMDE_FLOAT32_C(  -345.24), SIMDE_FLOAT32_C(    22.29),
        SIMDE_FLOAT32_C(    78.97), SIMDE_FLOAT32_C(   525.63), SIMDE_FLOAT32_C(   918.44), SIMDE_FLOAT32_C(  -210.86),
        SIMDE_FLOAT32_C(  -164.32), SIMDE_FLOAT32_C(   701.27), SIMDE_FLOAT32_C(   578.33), SIMDE_FLOAT32_C(  -261.40),
        SIMDE_FLOAT32_C(  -521.05), SIMDE_FLOAT32_C(  -156.56), SIMDE_FLOAT32_C(  -619.33), SIMDE_FLOAT32_C(   959.13) },
      { SIMDE_FLOAT32_C(  -115.03), SIMDE_FLOAT32_C(  -602.24), SIMDE_FLOAT32_C(   648.69), SIMDE_FLOAT32_C(   112.07) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(   265.11), SIMDE_FLOAT32_C(   -37.01), SIMDE_FLOAT32_C(  -345.24), SIMDE_FLOAT32_C(  -958.47),
        SIMDE_FLOAT32_C(  -982.91), SIMDE_FLOAT32_C(   525.63), SIMDE_FLOAT32_C(   918.44), SIMDE_FLOAT32_C(  -474.31),
        SIMDE_FLOAT32_C(  -164.32), SIMDE_FLOAT32_C(   701.27), SIMDE_FLOAT32_C(   578.33), SIMDE_FLOAT32_C(  -261.40),
        SIMDE_FLOAT32_C(  -115.03), SIMDE_FLOAT32_C(  -544.62), SIMDE_FLOAT32_C(   648.69), SIMDE_FLOAT32_C(   112.07) } },
    { { SIMDE_FLOAT32_C(  -890.15), SIMDE_FLOAT32_C(   781.68), SIMDE_FLOAT32_C(   329.66), SIMDE_FLOAT32_C(  -713.60),
        SIMDE_FLOAT32_C(   774.77), SIMDE_FLOAT32_C(   785.04), SIMDE_FLOAT32_C(  -609.14), SIMDE_FLOAT32_C(   -31.32),
        SIMDE_FLOAT32_C(  -575.55), SIMDE_FLOAT32_C(  -454.06), SIMDE_FLOAT32_C(   529.57), SIMDE_FLOAT32_C(   387.45),
        SIMDE_FLOAT32_C(   200.70), SIMDE_FLOAT32_C(  -448.14), SIMDE_FLOAT32_C(  -533.58), SIMDE_FLOAT32_C(  -273.68) },
      UINT16_C(59983),
      { SIMDE_FLOAT32_C(   562.01), SIMDE_FLOAT32_C(  -828.43), SIMDE_FLOAT32_C(  -166.11), SIMDE_FLOAT32_C(  -699.40),
        SIMDE_FLOAT32_C(  -349.48), SIMDE_FLOAT32_C(   677.33), SIMDE_FLOAT32_C(  -318.72), SIMDE_FLOAT32_C(  -390.35),
        SIMDE_FLOAT32_C(  -437.70), SIMDE_FLOAT32_C(    79.03), SIMDE_FLOAT32_C(  -741.67), SIMDE_FLOAT32_C(   674.37),
        SIMDE_FLOAT32_C(  -997.52), SIMDE_FLOAT32_C(  -631.82), SIMDE_FLOAT32_C(   456.05), SIMDE_FLOAT32_C(   332.15) },
      { SIMDE_FLOAT32_C(  -345.42), SIMDE_FLOAT32_C(   230.83), SIMDE_FLOAT32_C(   117.19), SIMDE_FLOAT32_C(    45.43) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(   562.01), SIMDE_FLOAT32_C(  -828.43), SIMDE_FLOAT32_C(  -166.11), SIMDE_FLOAT32_C(  -699.40),
        SIMDE_FLOAT32_C(   774.77), SIMDE_FLOAT32_C(   785.04), SIMDE_FLOAT32_C(  -318.72), SIMDE_FLOAT32_C(   -31.32),
        SIMDE_FLOAT32_C(  -575.55), SIMDE_FLOAT32_C(    79.03), SIMDE_FLOAT32_C(   529.57), SIMDE_FLOAT32_C(   674.37),
        SIMDE_FLOAT32_C(   200.70), SIMDE_FLOAT32_C(   230.83), SIMDE_FLOAT32_C(   117.19), SIMDE_FLOAT32_C(    45.43) } },
    { { SIMDE_FLOAT32_C(   541.64), SIMDE_FLOAT32_C(   591.37), SIMDE_FLOAT32_C(   729.08), SIMDE_FLOAT32_C(   -70.91),
        SIMDE_FLOAT32_C(  -207.93), SIMDE_FLOAT32_C(  -719.06), SIMDE_FLOAT32_C(   395.51), SIMDE_FLOAT32_C(   518.39),
        SIMDE_FLOAT32_C(  -248.76), SIMDE_FLOAT32_C(  -348.93), SIMDE_FLOAT32_C(    80.40), SIMDE_FLOAT32_C(   -77.19),
        SIMDE_FLOAT32_C(   484.96), SIMDE_FLOAT32_C(   381.00), SIMDE_FLOAT32_C(   573.32), SIMDE_FLOAT32_C(   162.29) },
      UINT16_C(55468),
      { SIMDE_FLOAT32_C(   724.60), SIMDE_FLOAT32_C(   141.31), SIMDE_FLOAT32_C(  -558.70), SIMDE_FLOAT32_C(   398.97),
        SIMDE_FLOAT32_C(   143.80), SIMDE_FLOAT32_C(  -190.52), SIMDE_FLOAT32_C(  -144.98), SIMDE_FLOAT32_C(  -524.06),
        SIMDE_FLOAT32_C(   464.06), SIMDE_FLOAT32_C(  -914.15), SIMDE_FLOAT32_C(   593.13), SIMDE_FLOAT32_C(  -490.51),
        SIMDE_FLOAT32_C(  -714.65), SIMDE_FLOAT32_C(   134.77), SIMDE_FLOAT32_C(  -899.13), SIMDE_FLOAT32_C(  -985.56) },
      { SIMDE_FLOAT32_C(  -936.14), SIMDE_FLOAT32_C(  -107.07), SIMDE_FLOAT32_C(  -704.62), SIMDE_FLOAT32_C(   459.37) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   541.64), SIMDE_FLOAT32_C(   591.37), SIMDE_FLOAT32_C(  -558.70), SIMDE_FLOAT32_C(   398.97),
        SIMDE_FLOAT32_C(  -207.93), SIMDE_FLOAT32_C(  -107.07), SIMDE_FLOAT32_C(   395.51), SIMDE_FLOAT32_C(   459.37),
        SIMDE_FLOAT32_C(  -248.76), SIMDE_FLOAT32_C(  -348.93), SIMDE_FLOAT32_C(    80.40), SIMDE_FLOAT32_C(  -490.51),
        SIMDE_FLOAT32_C(  -714.65), SIMDE_FLOAT32_C(   381.00), SIMDE_FLOAT32_C(  -899.13), SIMDE_FLOAT32_C(  -985.56) } },
    { { SIMDE_FLOAT32_C(    46.62), SIMDE_FLOAT32_C(  -889.56), SIMDE_FLOAT32_C(   491.72), SIMDE_FLOAT32_C(   969.43),
        SIMDE_FLOAT32_C(   595.40), SIMDE_FLOAT32_C(  -127.28), SIMDE_FLOAT32_C(   542.75), SIMDE_FLOAT32_C(  -242.30),
        SIMDE_FLOAT32_C(   -65.00), SIMDE_FLOAT32_C(   725.72), SIMDE_FLOAT32_C(  -517.71), SIMDE_FLOAT32_C(  -923.69),
        SIMDE_FLOAT32_C(  -832.98), SIMDE_FLOAT32_C(   881.26), SIMDE_FLOAT32_C(   220.11), SIMDE_FLOAT32_C(   -23.49) },
      UINT16_C(58625),
      { SIMDE_FLOAT32_C(  -559.43), SIMDE_FLOAT32_C(  -177.87), SIMDE_FLOAT32_C(   289.19), SIMDE_FLOAT32_C(   -49.93),
        SIMDE_FLOAT32_C(   107.48), SIMDE_FLOAT32_C(  -576.04), SIMDE_FLOAT32_C(    50.93), SIMDE_FLOAT32_C(   121.92),
        SIMDE_FLOAT32_C(  -512.17), SIMDE_FLOAT32_C(   943.86), SIMDE_FLOAT32_C(   417.30), SIMDE_FLOAT32_C(   947.20),
        SIMDE_FLOAT32_C(  -644.82), SIMDE_FLOAT32_C(  -536.08), SIMDE_FLOAT32_C(  -942.35), SIMDE_FLOAT32_C(   846.90) },
      { SIMDE_FLOAT32_C(  -566.65), SIMDE_FLOAT32_C(   653.05), SIMDE_FLOAT32_C(  -280.38), SIMDE_FLOAT32_C(   976.10) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(  -559.43), SIMDE_FLOAT32_C(  -889.56), SIMDE_FLOAT32_C(   491.72), SIMDE_FLOAT32_C(   969.43),
        SIMDE_FLOAT32_C(   595.40), SIMDE_FLOAT32_C(  -127.28), SIMDE_FLOAT32_C(   542.75), SIMDE_FLOAT32_C(  -242.30),
        SIMDE_FLOAT32_C(  -512.17), SIMDE_FLOAT32_C(   725.72), SIMDE_FLOAT32_C(   417.30), SIMDE_FLOAT32_C(  -923.69),
        SIMDE_FLOAT32_C(  -832.98), SIMDE_FLOAT32_C(  -536.08), SIMDE_FLOAT32_C(  -942.35), SIMDE_FLOAT32_C(   846.90) } },
    { { SIMDE_FLOAT32_C(   654.62), SIMDE_FLOAT32_C(   701.82), SIMDE_FLOAT32_C(  -106.96), SIMDE_FLOAT32_C(   730.93),
        SIMDE_FLOAT32_C(   868.84), SIMDE_FLOAT32_C(  -225.70), SIMDE_FLOAT32_C(   -48.95), SIMDE_FLOAT32_C(  -154.65),
        SIMDE_FLOAT32_C(   510.58), SIMDE_FLOAT32_C(  -352.90), SIMDE_FLOAT32_C(   285.92), SIMDE_FLOAT32_C(  -667.29),
        SIMDE_FLOAT32_C(   936.29), SIMDE_FLOAT32_C(  -764.01), SIMDE_FLOAT32_C(   440.19), SIMDE_FLOAT32_C(  -639.74) },
      UINT16_C(50560),
      { SIMDE_FLOAT32_C(  -151.91), SIMDE_FLOAT32_C(   230.78), SIMDE_FLOAT32_C(   979.41), SIMDE_FLOAT32_C(  -204.71),
        SIMDE_FLOAT32_C(   585.96), SIMDE_FLOAT32_C(  -556.67), SIMDE_FLOAT32_C(  -147.07), SIMDE_FLOAT32_C(   432.86),
        SIMDE_FLOAT32_C(  -123.32), SIMDE_FLOAT32_C(  -494.02), SIMDE_FLOAT32_C(  -847.52), SIMDE_FLOAT32_C(  -147.23),
        SIMDE_FLOAT32_C(   -83.27), SIMDE_FLOAT32_C(   807.10), SIMDE_FLOAT32_C(  -445.41), SIMDE_FLOAT32_C(   809.77) },
      { SIMDE_FLOAT32_C(   538.03), SIMDE_FLOAT32_C(  -576.56), SIMDE_FLOAT32_C(  -415.93), SIMDE_FLOAT32_C(  -510.92) },
       INT32_C(           2),
      { SIMDE_FLOAT32_C(   654.62), SIMDE_FLOAT32_C(   701.82), SIMDE_FLOAT32_C(  -106.96), SIMDE_FLOAT32_C(   730.93),
        SIMDE_FLOAT32_C(   868.84), SIMDE_FLOAT32_C(  -225.70), SIMDE_FLOAT32_C(   -48.95), SIMDE_FLOAT32_C(   432.86),
        SIMDE_FLOAT32_C(   538.03), SIMDE_FLOAT32_C(  -352.90), SIMDE_FLOAT32_C(  -415.93), SIMDE_FLOAT32_C(  -667.29),
        SIMDE_FLOAT32_C(   936.29), SIMDE_FLOAT32_C(  -764.01), SIMDE_FLOAT32_C(  -445.41), SIMDE_FLOAT32_C(   809.77) } },
    { { SIMDE_FLOAT32_C(  -905.35), SIMDE_FLOAT32_C(   136.18), SIMDE_FLOAT32_C(  -445.28), SIMDE_FLOAT32_C(  -572.64),
        SIMDE_FLOAT32_C(    72.48), SIMDE_FLOAT32_C(  -209.30), SIMDE_FLOAT32_C(   867.55), SIMDE_FLOAT32_C(   432.73),
        SIMDE_FLOAT32_C(  -922.38), SIMDE_FLOAT32_C(  -570.33), SIMDE_FLOAT32_C(  -719.18), SIMDE_FLOAT32_C(   308.40),
        SIMDE_FLOAT32_C(  -590.92), SIMDE_FLOAT32_C(    76.11), SIMDE_FLOAT32_C(  -105.64), SIMDE_FLOAT32_C(  -147.59) },
      UINT16_C(25801),
      { SIMDE_FLOAT32_C(   729.08), SIMDE_FLOAT32_C(  -564.98), SIMDE_FLOAT32_C(  -520.30), SIMDE_FLOAT32_C(  -418.14),
        SIMDE_FLOAT32_C(   351.75), SIMDE_FLOAT32_C(  -713.20), SIMDE_FLOAT32_C(   136.45), SIMDE_FLOAT32_C(   161.52),
        SIMDE_FLOAT32_C(   824.83), SIMDE_FLOAT32_C(   559.89), SIMDE_FLOAT32_C(   745.58), SIMDE_FLOAT32_C(  -686.09),
        SIMDE_FLOAT32_C(  -171.32), SIMDE_FLOAT32_C(   840.23), SIMDE_FLOAT32_C(   450.09), SIMDE_FLOAT32_C(   383.40) },
      { SIMDE_FLOAT32_C(  -732.41), SIMDE_FLOAT32_C(  -477.43), SIMDE_FLOAT32_C(  -825.90), SIMDE_FLOAT32_C(  -864.86) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   729.08), SIMDE_FLOAT32_C(   136.18), SIMDE_FLOAT32_C(  -445.28), SIMDE_FLOAT32_C(  -418.14),
        SIMDE_FLOAT32_C(    72.48), SIMDE_FLOAT32_C(  -209.30), SIMDE_FLOAT32_C(  -825.90), SIMDE_FLOAT32_C(  -864.86),
        SIMDE_FLOAT32_C(  -922.38), SIMDE_FLOAT32_C(  -570.33), SIMDE_FLOAT32_C(   745.58), SIMDE_FLOAT32_C(   308.40),
        SIMDE_FLOAT32_C(  -590.92), SIMDE_FLOAT32_C(   840.23), SIMDE_FLOAT32_C(   450.09), SIMDE_FLOAT32_C(  -147.59) } },
    { { SIMDE_FLOAT32_C(  -748.28), SIMDE_FLOAT32_C(  -435.19), SIMDE_FLOAT32_C(  -763.88), SIMDE_FLOAT32_C(   560.12),
        SIMDE_FLOAT32_C(   -26.11), SIMDE_FLOAT32_C(   312.23), SIMDE_FLOAT32_C(  -545.52), SIMDE_FLOAT32_C(   826.30),
        SIMDE_FLOAT32_C(   241.27), SIMDE_FLOAT32_C(  -218.30), SIMDE_FLOAT32_C(   555.38), SIMDE_FLOAT32_C(   676.29),
        SIMDE_FLOAT32_C(   261.40), SIMDE_FLOAT32_C(  -862.76), SIMDE_FLOAT32_C(    28.04), SIMDE_FLOAT32_C(   548.20) },
      UINT16_C(39347),
      { SIMDE_FLOAT32_C(   373.02), SIMDE_FLOAT32_C(  -166.42), SIMDE_FLOAT32_C(   935.14), SIMDE_FLOAT32_C(   686.93),
        SIMDE_FLOAT32_C(   662.27), SIMDE_FLOAT32_C(   775.37), SIMDE_FLOAT32_C(   137.02), SIMDE_FLOAT32_C(    45.66),
        SIMDE_FLOAT32_C(  -957.05), SIMDE_FLOAT32_C(   659.59), SIMDE_FLOAT32_C(   219.77), SIMDE_FLOAT32_C(  -821.90),
        SIMDE_FLOAT32_C(   614.89), SIMDE_FLOAT32_C(   471.49), SIMDE_FLOAT32_C(  -257.09), SIMDE_FLOAT32_C(   851.01) },
      { SIMDE_FLOAT32_C(    31.61), SIMDE_FLOAT32_C(   716.80), SIMDE_FLOAT32_C(   163.24), SIMDE_FLOAT32_C(   486.10) },
       INT32_C(           1),
      { SIMDE_FLOAT32_C(   373.02), SIMDE_FLOAT32_C(  -166.42), SIMDE_FLOAT32_C(  -763.88), SIMDE_FLOAT32_C(   560.12),
        SIMDE_FLOAT32_C(    31.61), SIMDE_FLOAT32_C(   716.80), SIMDE_FLOAT32_C(  -545.52), SIMDE_FLOAT32_C(   486.10),
        SIMDE_FLOAT32_C(  -957.05), SIMDE_FLOAT32_C(  -218.30), SIMDE_FLOAT32_C(   555.38), SIMDE_FLOAT32_C(  -821.90),
        SIMDE_FLOAT32_C(   614.89), SIMDE_FLOAT32_C(  -862.76), SIMDE_FLOAT32_C(    28.04), SIMDE_FLOAT32_C(   851.01) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 src = simde_mm512_loadu_ps(test_vec[i].src);
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m128 b = simde_mm_loadu_ps(test_vec[i].b);
    simde__m512 r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_insertf32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), test_vec[i].imm8, src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512 src = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m128 b = simde_test_x86_random_f32x4(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 3;
    simde__m512 r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_insertf32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), imm8, src, k, a, b);

    simde_test_x86_write_f32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_insertf32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const float a[16];
    const float b[4];
    const int imm8;
    const float r[16];
  } test_vec[] = {
    { UINT16_C(40380),
      { SIMDE_FLOAT32_C(   697.69), SIMDE_FLOAT32_C(   200.65), SIMDE_FLOAT32_C(   426.17), SIMDE_FLOAT32_C(   249.04),
        SIMDE_FLOAT32_C(  -279.29), SIMDE_FLOAT32_C(  -148.87), SIMDE_FLOAT32_C(   862.73), SIMDE_FLOAT32_C(   413.25),
        SIMDE_FLOAT32_C(  -424.28), SIMDE_FLOAT32_C(  -997.87), SIMDE_FLOAT32_C(   966.84), SIMDE_FLOAT32_C(   266.13),
        SIMDE_FLOAT32_C(   105.29), SIMDE_FLOAT32_C(  -474.25), SIMDE_FLOAT32_C(   522.51), SIMDE_FLOAT32_C(   401.25) },
      { SIMDE_FLOAT32_C(  -478.02), SIMDE_FLOAT32_C(  -353.44), SIMDE_FLOAT32_C(   717.92), SIMDE_FLOAT32_C(   307.82) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   717.92), SIMDE_FLOAT32_C(   307.82),
        SIMDE_FLOAT32_C(  -279.29), SIMDE_FLOAT32_C(  -148.87), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   413.25),
        SIMDE_FLOAT32_C(  -424.28), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   966.84), SIMDE_FLOAT32_C(   266.13),
        SIMDE_FLOAT32_C(   105.29), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   401.25) } },
    { UINT16_C(22454),
      { SIMDE_FLOAT32_C(   492.78), SIMDE_FLOAT32_C(  -365.48), SIMDE_FLOAT32_C(   431.24), SIMDE_FLOAT32_C(  -312.01),
        SIMDE_FLOAT32_C(  -581.53), SIMDE_FLOAT32_C(  -328.62), SIMDE_FLOAT32_C(  -946.17), SIMDE_FLOAT32_C(  -326.78),
        SIMDE_FLOAT32_C(  -630.92), SIMDE_FLOAT32_C(   254.49), SIMDE_FLOAT32_C(  -900.61), SIMDE_FLOAT32_C(   618.12),
        SIMDE_FLOAT32_C(   975.20), SIMDE_FLOAT32_C(   -49.48), SIMDE_FLOAT32_C(   480.85), SIMDE_FLOAT32_C(   388.44) },
      { SIMDE_FLOAT32_C(   526.24), SIMDE_FLOAT32_C(   482.97), SIMDE_FLOAT32_C(   355.28), SIMDE_FLOAT32_C(  -207.64) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   482.97), SIMDE_FLOAT32_C(   355.28), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -581.53), SIMDE_FLOAT32_C(  -328.62), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -326.78),
        SIMDE_FLOAT32_C(  -630.92), SIMDE_FLOAT32_C(   254.49), SIMDE_FLOAT32_C(  -900.61), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   975.20), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   480.85), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(56983),
      { SIMDE_FLOAT32_C(   989.52), SIMDE_FLOAT32_C(  -596.99), SIMDE_FLOAT32_C(   -38.57), SIMDE_FLOAT32_C(   707.44),
        SIMDE_FLOAT32_C(   710.83), SIMDE_FLOAT32_C(  -335.90), SIMDE_FLOAT32_C(  -785.18), SIMDE_FLOAT32_C(   262.50),
        SIMDE_FLOAT32_C(  -843.12), SIMDE_FLOAT32_C(  -150.66), SIMDE_FLOAT32_C(  -306.26), SIMDE_FLOAT32_C(  -155.13),
        SIMDE_FLOAT32_C(   267.81), SIMDE_FLOAT32_C(   365.13), SIMDE_FLOAT32_C(  -101.29), SIMDE_FLOAT32_C(   941.03) },
      { SIMDE_FLOAT32_C(   734.21), SIMDE_FLOAT32_C(  -846.81), SIMDE_FLOAT32_C(  -959.59), SIMDE_FLOAT32_C(   352.32) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(   989.52), SIMDE_FLOAT32_C(  -596.99), SIMDE_FLOAT32_C(   -38.57), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   710.83), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   262.50),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -150.66), SIMDE_FLOAT32_C(  -306.26), SIMDE_FLOAT32_C(  -155.13),
        SIMDE_FLOAT32_C(   734.21), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -959.59), SIMDE_FLOAT32_C(   352.32) } },
    { UINT16_C(27577),
      { SIMDE_FLOAT32_C(   516.83), SIMDE_FLOAT32_C(  -482.84), SIMDE_FLOAT32_C(  -683.85), SIMDE_FLOAT32_C(  -127.88),
        SIMDE_FLOAT32_C(   309.53), SIMDE_FLOAT32_C(   -95.59), SIMDE_FLOAT32_C(  -246.85), SIMDE_FLOAT32_C(   624.40),
        SIMDE_FLOAT32_C(  -106.07), SIMDE_FLOAT32_C(   156.16), SIMDE_FLOAT32_C(  -414.17), SIMDE_FLOAT32_C(  -398.64),
        SIMDE_FLOAT32_C(  -133.01), SIMDE_FLOAT32_C(   249.93), SIMDE_FLOAT32_C(  -183.81), SIMDE_FLOAT32_C(  -870.51) },
      { SIMDE_FLOAT32_C(   406.81), SIMDE_FLOAT32_C(   665.53), SIMDE_FLOAT32_C(  -176.76), SIMDE_FLOAT32_C(  -748.32) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(   516.83), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -127.88),
        SIMDE_FLOAT32_C(   309.53), SIMDE_FLOAT32_C(   -95.59), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   624.40),
        SIMDE_FLOAT32_C(  -106.07), SIMDE_FLOAT32_C(   156.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -398.64),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   665.53), SIMDE_FLOAT32_C(  -176.76), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(62379),
      { SIMDE_FLOAT32_C(  -125.64), SIMDE_FLOAT32_C(   922.57), SIMDE_FLOAT32_C(   303.59), SIMDE_FLOAT32_C(   -85.23),
        SIMDE_FLOAT32_C(   274.89), SIMDE_FLOAT32_C(   431.98), SIMDE_FLOAT32_C(   905.70), SIMDE_FLOAT32_C(  -891.94),
        SIMDE_FLOAT32_C(   -51.19), SIMDE_FLOAT32_C(  -577.14), SIMDE_FLOAT32_C(  -575.79), SIMDE_FLOAT32_C(   820.93),
        SIMDE_FLOAT32_C(   732.39), SIMDE_FLOAT32_C(   328.62), SIMDE_FLOAT32_C(  -425.93), SIMDE_FLOAT32_C(   356.79) },
      { SIMDE_FLOAT32_C(  -777.45), SIMDE_FLOAT32_C(   730.23), SIMDE_FLOAT32_C(   942.61), SIMDE_FLOAT32_C(  -176.08) },
       INT32_C(           2),
      { SIMDE_FLOAT32_C(  -125.64), SIMDE_FLOAT32_C(   922.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -85.23),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   431.98), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -891.94),
        SIMDE_FLOAT32_C(  -777.45), SIMDE_FLOAT32_C(   730.23), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   732.39), SIMDE_FLOAT32_C(   328.62), SIMDE_FLOAT32_C(  -425.93), SIMDE_FLOAT32_C(   356.79) } },
    { UINT16_C( 8569),
      { SIMDE_FLOAT32_C(  -273.29), SIMDE_FLOAT32_C(  -400.65), SIMDE_FLOAT32_C(   305.63), SIMDE_FLOAT32_C(   549.95),
        SIMDE_FLOAT32_C(  -148.96), SIMDE_FLOAT32_C(  -761.04), SIMDE_FLOAT32_C(   738.31), SIMDE_FLOAT32_C(  -998.57),
        SIMDE_FLOAT32_C(   113.32), SIMDE_FLOAT32_C(   660.88), SIMDE_FLOAT32_C(   305.02), SIMDE_FLOAT32_C(  -971.91),
        SIMDE_FLOAT32_C(   -64.23), SIMDE_FLOAT32_C(  -263.01), SIMDE_FLOAT32_C(   933.79), SIMDE_FLOAT32_C(    43.84) },
      { SIMDE_FLOAT32_C(   685.80), SIMDE_FLOAT32_C(  -643.35), SIMDE_FLOAT32_C(   468.05), SIMDE_FLOAT32_C(   506.73) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(  -273.29), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   549.95),
        SIMDE_FLOAT32_C(  -148.96), SIMDE_FLOAT32_C(  -761.04), SIMDE_FLOAT32_C(   738.31), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   113.32), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -643.35), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(35711),
      { SIMDE_FLOAT32_C(   445.83), SIMDE_FLOAT32_C(    19.22), SIMDE_FLOAT32_C(   811.03), SIMDE_FLOAT32_C(   388.44),
        SIMDE_FLOAT32_C(   843.14), SIMDE_FLOAT32_C(  -591.75), SIMDE_FLOAT32_C(  -419.02), SIMDE_FLOAT32_C(   483.24),
        SIMDE_FLOAT32_C(   134.96), SIMDE_FLOAT32_C(   180.33), SIMDE_FLOAT32_C(  -211.13), SIMDE_FLOAT32_C(  -315.09),
        SIMDE_FLOAT32_C(  -968.63), SIMDE_FLOAT32_C(    27.83), SIMDE_FLOAT32_C(  -576.78), SIMDE_FLOAT32_C(  -967.20) },
      { SIMDE_FLOAT32_C(  -858.85), SIMDE_FLOAT32_C(  -915.90), SIMDE_FLOAT32_C(   337.81), SIMDE_FLOAT32_C(  -830.76) },
       INT32_C(           0),
      { SIMDE_FLOAT32_C(  -858.85), SIMDE_FLOAT32_C(  -915.90), SIMDE_FLOAT32_C(   337.81), SIMDE_FLOAT32_C(  -830.76),
        SIMDE_FLOAT32_C(   843.14), SIMDE_FLOAT32_C(  -591.75), SIMDE_FLOAT32_C(  -419.02), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   134.96), SIMDE_FLOAT32_C(   180.33), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -315.09),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -967.20) } },
    { UINT16_C(61371),
      { SIMDE_FLOAT32_C(  -936.29), SIMDE_FLOAT32_C(   760.61), SIMDE_FLOAT32_C(  -540.32), SIMDE_FLOAT32_C(   531.76),
        SIMDE_FLOAT32_C(   267.34), SIMDE_FLOAT32_C(  -451.28), SIMDE_FLOAT32_C(  -671.57), SIMDE_FLOAT32_C(   348.14),
        SIMDE_FLOAT32_C(   994.54), SIMDE_FLOAT32_C(   347.65), SIMDE_FLOAT32_C(   159.18), SIMDE_FLOAT32_C(   382.98),
        SIMDE_FLOAT32_C(   190.79), SIMDE_FLOAT32_C(   567.43), SIMDE_FLOAT32_C(   963.96), SIMDE_FLOAT32_C(  -325.96) },
      { SIMDE_FLOAT32_C(  -297.61), SIMDE_FLOAT32_C(   144.29), SIMDE_FLOAT32_C(   462.91), SIMDE_FLOAT32_C(   387.30) },
       INT32_C(           3),
      { SIMDE_FLOAT32_C(  -936.29), SIMDE_FLOAT32_C(   760.61), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   531.76),
        SIMDE_FLOAT32_C(   267.34), SIMDE_FLOAT32_C(  -451.28), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   348.14),
        SIMDE_FLOAT32_C(   994.54), SIMDE_FLOAT32_C(   347.65), SIMDE_FLOAT32_C(   159.18), SIMDE_FLOAT32_C(   382.98),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   144.29), SIMDE_FLOAT32_C(   462.91), SIMDE_FLOAT32_C(   387.30) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m128 b = simde_mm_loadu_ps(test_vec[i].b);
    simde__m512 r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_insertf32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), test_vec[i].imm8, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m128 b = simde_test_x86_random_f32x4(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 3;
    simde__m512 r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_insertf32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), imm8, k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_insertf64x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const double a[8];
    const double b[4];
    const int imm8;
    const double r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   790.28), SIMDE_FLOAT64_C(   -27.98), SIMDE_FLOAT64_C(   336.38), SIMDE_FLOAT64_C(   192.28),
        SIMDE_FLOAT64_C(  -777.90), SIMDE_FLOAT64_C(   219.30), SIMDE_FLOAT64_C(   915.82), SIMDE_FLOAT64_C(   246.17) },
      { SIMDE_FLOAT64_C(   -41.72), SIMDE_FLOAT64_C(   403.90), SIMDE_FLOAT64_C(  -421.32), SIMDE_FLOAT64_C(   487.02) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   790.28), SIMDE_FLOAT64_C(   -27.98), SIMDE_FLOAT64_C(   336.38), SIMDE_FLOAT64_C(   192.28),
        SIMDE_FLOAT64_C(   -41.72), SIMDE_FLOAT64_C(   403.90), SIMDE_FLOAT64_C(  -421.32), SIMDE_FLOAT64_C(   487.02) } },
    { { SIMDE_FLOAT64_C(   220.73), SIMDE_FLOAT64_C(   505.67), SIMDE_FLOAT64_C(  -666.23), SIMDE_FLOAT64_C(   332.24),
        SIMDE_FLOAT64_C(  -874.94), SIMDE_FLOAT64_C(   550.34), SIMDE_FLOAT64_C(   114.23), SIMDE_FLOAT64_C(  -368.42) },
      { SIMDE_FLOAT64_C(  -553.17), SIMDE_FLOAT64_C(   451.88), SIMDE_FLOAT64_C(   831.65), SIMDE_FLOAT64_C(   177.91) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   220.73), SIMDE_FLOAT64_C(   505.67), SIMDE_FLOAT64_C(  -666.23), SIMDE_FLOAT64_C(   332.24),
        SIMDE_FLOAT64_C(  -553.17), SIMDE_FLOAT64_C(   451.88), SIMDE_FLOAT64_C(   831.65), SIMDE_FLOAT64_C(   177.91) } },
    { { SIMDE_FLOAT64_C(   307.58), SIMDE_FLOAT64_C(   528.28), SIMDE_FLOAT64_C(  -980.41), SIMDE_FLOAT64_C(   736.66),
        SIMDE_FLOAT64_C(   391.46), SIMDE_FLOAT64_C(   809.86), SIMDE_FLOAT64_C(  -291.32), SIMDE_FLOAT64_C(  -272.16) },
      { SIMDE_FLOAT64_C(     2.14), SIMDE_FLOAT64_C(   -69.22), SIMDE_FLOAT64_C(   947.15), SIMDE_FLOAT64_C(   -82.04) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(     2.14), SIMDE_FLOAT64_C(   -69.22), SIMDE_FLOAT64_C(   947.15), SIMDE_FLOAT64_C(   -82.04),
        SIMDE_FLOAT64_C(   391.46), SIMDE_FLOAT64_C(   809.86), SIMDE_FLOAT64_C(  -291.32), SIMDE_FLOAT64_C(  -272.16) } },
    { { SIMDE_FLOAT64_C(   -94.57), SIMDE_FLOAT64_C(  -678.14), SIMDE_FLOAT64_C(  -244.38), SIMDE_FLOAT64_C(  -607.55),
        SIMDE_FLOAT64_C(   539.70), SIMDE_FLOAT64_C(   976.35), SIMDE_FLOAT64_C(   898.11), SIMDE_FLOAT64_C(   873.47) },
      { SIMDE_FLOAT64_C(   308.59), SIMDE_FLOAT64_C(  -976.83), SIMDE_FLOAT64_C(   423.81), SIMDE_FLOAT64_C(  -577.18) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   308.59), SIMDE_FLOAT64_C(  -976.83), SIMDE_FLOAT64_C(   423.81), SIMDE_FLOAT64_C(  -577.18),
        SIMDE_FLOAT64_C(   539.70), SIMDE_FLOAT64_C(   976.35), SIMDE_FLOAT64_C(   898.11), SIMDE_FLOAT64_C(   873.47) } },
    { { SIMDE_FLOAT64_C(   870.64), SIMDE_FLOAT64_C(   874.69), SIMDE_FLOAT64_C(  -513.60), SIMDE_FLOAT64_C(    48.54),
        SIMDE_FLOAT64_C(   410.26), SIMDE_FLOAT64_C(   793.99), SIMDE_FLOAT64_C(  -423.18), SIMDE_FLOAT64_C(   429.85) },
      { SIMDE_FLOAT64_C(   530.65), SIMDE_FLOAT64_C(   968.28), SIMDE_FLOAT64_C(   239.71), SIMDE_FLOAT64_C(  -760.67) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   530.65), SIMDE_FLOAT64_C(   968.28), SIMDE_FLOAT64_C(   239.71), SIMDE_FLOAT64_C(  -760.67),
        SIMDE_FLOAT64_C(   410.26), SIMDE_FLOAT64_C(   793.99), SIMDE_FLOAT64_C(  -423.18), SIMDE_FLOAT64_C(   429.85) } },
    { { SIMDE_FLOAT64_C(  -758.15), SIMDE_FLOAT64_C(   170.10), SIMDE_FLOAT64_C(  -356.73), SIMDE_FLOAT64_C(   159.81),
        SIMDE_FLOAT64_C(   347.05), SIMDE_FLOAT64_C(   548.69), SIMDE_FLOAT64_C(   481.66), SIMDE_FLOAT64_C(  -897.33) },
      { SIMDE_FLOAT64_C(   941.14), SIMDE_FLOAT64_C(    21.36), SIMDE_FLOAT64_C(  -920.98), SIMDE_FLOAT64_C(   839.25) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -758.15), SIMDE_FLOAT64_C(   170.10), SIMDE_FLOAT64_C(  -356.73), SIMDE_FLOAT64_C(   159.81),
        SIMDE_FLOAT64_C(   941.14), SIMDE_FLOAT64_C(    21.36), SIMDE_FLOAT64_C(  -920.98), SIMDE_FLOAT64_C(   839.25) } },
    { { SIMDE_FLOAT64_C(   387.61), SIMDE_FLOAT64_C(   862.42), SIMDE_FLOAT64_C(  -681.36), SIMDE_FLOAT64_C(   810.42),
        SIMDE_FLOAT64_C(  -482.83), SIMDE_FLOAT64_C(  -810.72), SIMDE_FLOAT64_C(   685.12), SIMDE_FLOAT64_C(     3.57) },
      { SIMDE_FLOAT64_C(   237.82), SIMDE_FLOAT64_C(    95.38), SIMDE_FLOAT64_C(  -202.44), SIMDE_FLOAT64_C(   814.64) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   387.61), SIMDE_FLOAT64_C(   862.42), SIMDE_FLOAT64_C(  -681.36), SIMDE_FLOAT64_C(   810.42),
        SIMDE_FLOAT64_C(   237.82), SIMDE_FLOAT64_C(    95.38), SIMDE_FLOAT64_C(  -202.44), SIMDE_FLOAT64_C(   814.64) } },
    { { SIMDE_FLOAT64_C(  -671.79), SIMDE_FLOAT64_C(   782.92), SIMDE_FLOAT64_C(   764.93), SIMDE_FLOAT64_C(  -432.47),
        SIMDE_FLOAT64_C(  -520.95), SIMDE_FLOAT64_C(  -993.22), SIMDE_FLOAT64_C(   737.64), SIMDE_FLOAT64_C(   122.32) },
      { SIMDE_FLOAT64_C(   166.59), SIMDE_FLOAT64_C(    84.68), SIMDE_FLOAT64_C(  -328.99), SIMDE_FLOAT64_C(  -351.75) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   166.59), SIMDE_FLOAT64_C(    84.68), SIMDE_FLOAT64_C(  -328.99), SIMDE_FLOAT64_C(  -351.75),
        SIMDE_FLOAT64_C(  -520.95), SIMDE_FLOAT64_C(  -993.22), SIMDE_FLOAT64_C(   737.64), SIMDE_FLOAT64_C(   122.32) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m256d b = simde_mm256_loadu_pd(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_f64x8(simde_mm512_insertf64x4(a, b, 0), simde_mm512_loadu_pd(test_vec[i].r), 1); break;
      case 1: simde_test_x86_assert_equal_f64x8(simde_mm512_insertf64x4(a, b, 1), simde_mm512_loadu_pd(test_vec[i].r), 1); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m256d b = simde_test_x86_random_f64x4(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m512d r;
    SIMDE_CONSTIFY_2_(simde_mm512_insertf64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), imm8, a, b);

    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_insertf64x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const double src[8];
    const simde__mmask8 k;
    const double a[8];
    const double b[4];
    const int imm8;
    const double r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -387.85), SIMDE_FLOAT64_C(   669.62), SIMDE_FLOAT64_C(   266.37), SIMDE_FLOAT64_C(  -548.60),
        SIMDE_FLOAT64_C(  -435.55), SIMDE_FLOAT64_C(  -346.03), SIMDE_FLOAT64_C(  -686.18), SIMDE_FLOAT64_C(  -116.90) },
      UINT8_C(210),
      { SIMDE_FLOAT64_C(  -169.01), SIMDE_FLOAT64_C(    72.38), SIMDE_FLOAT64_C(  -850.49), SIMDE_FLOAT64_C(   834.56),
        SIMDE_FLOAT64_C(  -689.80), SIMDE_FLOAT64_C(   244.89), SIMDE_FLOAT64_C(  -367.89), SIMDE_FLOAT64_C(  -875.15) },
      { SIMDE_FLOAT64_C(   770.12), SIMDE_FLOAT64_C(   -39.68), SIMDE_FLOAT64_C(   907.77), SIMDE_FLOAT64_C(   535.05) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -387.85), SIMDE_FLOAT64_C(    72.38), SIMDE_FLOAT64_C(   266.37), SIMDE_FLOAT64_C(  -548.60),
        SIMDE_FLOAT64_C(   770.12), SIMDE_FLOAT64_C(  -346.03), SIMDE_FLOAT64_C(   907.77), SIMDE_FLOAT64_C(   535.05) } },
    { { SIMDE_FLOAT64_C(  -613.18), SIMDE_FLOAT64_C(   541.83), SIMDE_FLOAT64_C(   265.49), SIMDE_FLOAT64_C(   509.13),
        SIMDE_FLOAT64_C(  -291.58), SIMDE_FLOAT64_C(  -649.83), SIMDE_FLOAT64_C(  -819.86), SIMDE_FLOAT64_C(   356.68) },
      UINT8_C(  5),
      { SIMDE_FLOAT64_C(  -207.71), SIMDE_FLOAT64_C(    26.29), SIMDE_FLOAT64_C(  -196.11), SIMDE_FLOAT64_C(   243.70),
        SIMDE_FLOAT64_C(   590.75), SIMDE_FLOAT64_C(   457.86), SIMDE_FLOAT64_C(   557.52), SIMDE_FLOAT64_C(  -526.15) },
      { SIMDE_FLOAT64_C(   922.26), SIMDE_FLOAT64_C(  -611.50), SIMDE_FLOAT64_C(   546.22), SIMDE_FLOAT64_C(  -928.23) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   922.26), SIMDE_FLOAT64_C(   541.83), SIMDE_FLOAT64_C(   546.22), SIMDE_FLOAT64_C(   509.13),
        SIMDE_FLOAT64_C(  -291.58), SIMDE_FLOAT64_C(  -649.83), SIMDE_FLOAT64_C(  -819.86), SIMDE_FLOAT64_C(   356.68) } },
    { { SIMDE_FLOAT64_C(   856.43), SIMDE_FLOAT64_C(   316.66), SIMDE_FLOAT64_C(  -144.83), SIMDE_FLOAT64_C(   981.27),
        SIMDE_FLOAT64_C(    86.78), SIMDE_FLOAT64_C(   815.49), SIMDE_FLOAT64_C(   889.04), SIMDE_FLOAT64_C(  -378.17) },
      UINT8_C(196),
      { SIMDE_FLOAT64_C(  -724.14), SIMDE_FLOAT64_C(  -836.34), SIMDE_FLOAT64_C(  -391.16), SIMDE_FLOAT64_C(   784.99),
        SIMDE_FLOAT64_C(  -127.92), SIMDE_FLOAT64_C(   -40.99), SIMDE_FLOAT64_C(   965.14), SIMDE_FLOAT64_C(  -771.24) },
      { SIMDE_FLOAT64_C(  -503.46), SIMDE_FLOAT64_C(  -242.57), SIMDE_FLOAT64_C(   255.05), SIMDE_FLOAT64_C(   300.42) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   856.43), SIMDE_FLOAT64_C(   316.66), SIMDE_FLOAT64_C(   255.05), SIMDE_FLOAT64_C(   981.27),
        SIMDE_FLOAT64_C(    86.78), SIMDE_FLOAT64_C(   815.49), SIMDE_FLOAT64_C(   965.14), SIMDE_FLOAT64_C(  -771.24) } },
    { { SIMDE_FLOAT64_C(  -154.20), SIMDE_FLOAT64_C(  -241.71), SIMDE_FLOAT64_C(   558.64), SIMDE_FLOAT64_C(   319.64),
        SIMDE_FLOAT64_C(  -319.46), SIMDE_FLOAT64_C(   947.15), SIMDE_FLOAT64_C(  -134.13), SIMDE_FLOAT64_C(  -247.69) },
      UINT8_C(188),
      { SIMDE_FLOAT64_C(  -277.71), SIMDE_FLOAT64_C(  -931.03), SIMDE_FLOAT64_C(    25.37), SIMDE_FLOAT64_C(  -296.44),
        SIMDE_FLOAT64_C(   155.75), SIMDE_FLOAT64_C(  -159.14), SIMDE_FLOAT64_C(  -407.39), SIMDE_FLOAT64_C(   777.57) },
      { SIMDE_FLOAT64_C(  -815.79), SIMDE_FLOAT64_C(  -131.54), SIMDE_FLOAT64_C(   941.23), SIMDE_FLOAT64_C(  -206.95) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -154.20), SIMDE_FLOAT64_C(  -241.71), SIMDE_FLOAT64_C(    25.37), SIMDE_FLOAT64_C(  -296.44),
        SIMDE_FLOAT64_C(  -815.79), SIMDE_FLOAT64_C(  -131.54), SIMDE_FLOAT64_C(  -134.13), SIMDE_FLOAT64_C(  -206.95) } },
    { { SIMDE_FLOAT64_C(  -186.69), SIMDE_FLOAT64_C(   752.06), SIMDE_FLOAT64_C(  -381.41), SIMDE_FLOAT64_C(    42.07),
        SIMDE_FLOAT64_C(  -751.41), SIMDE_FLOAT64_C(   376.02), SIMDE_FLOAT64_C(  -702.88), SIMDE_FLOAT64_C(   549.02) },
      UINT8_C( 76),
      { SIMDE_FLOAT64_C(   142.92), SIMDE_FLOAT64_C(  -692.70), SIMDE_FLOAT64_C(   -64.20), SIMDE_FLOAT64_C(  -537.43),
        SIMDE_FLOAT64_C(   -12.16), SIMDE_FLOAT64_C(  -117.06), SIMDE_FLOAT64_C(   328.43), SIMDE_FLOAT64_C(   740.15) },
      { SIMDE_FLOAT64_C(    53.14), SIMDE_FLOAT64_C(  -949.27), SIMDE_FLOAT64_C(   809.12), SIMDE_FLOAT64_C(  -921.48) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(  -186.69), SIMDE_FLOAT64_C(   752.06), SIMDE_FLOAT64_C(   809.12), SIMDE_FLOAT64_C(  -921.48),
        SIMDE_FLOAT64_C(  -751.41), SIMDE_FLOAT64_C(   376.02), SIMDE_FLOAT64_C(   328.43), SIMDE_FLOAT64_C(   549.02) } },
    { { SIMDE_FLOAT64_C(   -35.13), SIMDE_FLOAT64_C(   -80.62), SIMDE_FLOAT64_C(   346.89), SIMDE_FLOAT64_C(  -257.56),
        SIMDE_FLOAT64_C(   103.59), SIMDE_FLOAT64_C(  -784.64), SIMDE_FLOAT64_C(  -316.32), SIMDE_FLOAT64_C(   896.64) },
      UINT8_C( 16),
      { SIMDE_FLOAT64_C(   496.99), SIMDE_FLOAT64_C(   648.69), SIMDE_FLOAT64_C(   487.41), SIMDE_FLOAT64_C(  -460.94),
        SIMDE_FLOAT64_C(   897.29), SIMDE_FLOAT64_C(  -136.57), SIMDE_FLOAT64_C(  -163.82), SIMDE_FLOAT64_C(   446.30) },
      { SIMDE_FLOAT64_C(  -759.42), SIMDE_FLOAT64_C(   979.10), SIMDE_FLOAT64_C(   753.60), SIMDE_FLOAT64_C(   176.38) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   -35.13), SIMDE_FLOAT64_C(   -80.62), SIMDE_FLOAT64_C(   346.89), SIMDE_FLOAT64_C(  -257.56),
        SIMDE_FLOAT64_C(  -759.42), SIMDE_FLOAT64_C(  -784.64), SIMDE_FLOAT64_C(  -316.32), SIMDE_FLOAT64_C(   896.64) } },
    { { SIMDE_FLOAT64_C(  -258.55), SIMDE_FLOAT64_C(  -940.68), SIMDE_FLOAT64_C(   770.10), SIMDE_FLOAT64_C(  -518.40),
        SIMDE_FLOAT64_C(   112.47), SIMDE_FLOAT64_C(   820.83), SIMDE_FLOAT64_C(  -709.28), SIMDE_FLOAT64_C(   190.99) },
      UINT8_C(166),
      { SIMDE_FLOAT64_C(   255.59), SIMDE_FLOAT64_C(  -889.63), SIMDE_FLOAT64_C(   922.01), SIMDE_FLOAT64_C(   998.04),
        SIMDE_FLOAT64_C(   213.96), SIMDE_FLOAT64_C(  -862.63), SIMDE_FLOAT64_C(  -318.28), SIMDE_FLOAT64_C(   110.60) },
      { SIMDE_FLOAT64_C(     6.19), SIMDE_FLOAT64_C(  -821.29), SIMDE_FLOAT64_C(  -240.70), SIMDE_FLOAT64_C(  -506.40) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -258.55), SIMDE_FLOAT64_C(  -889.63), SIMDE_FLOAT64_C(   922.01), SIMDE_FLOAT64_C(  -518.40),
        SIMDE_FLOAT64_C(   112.47), SIMDE_FLOAT64_C(  -821.29), SIMDE_FLOAT64_C(  -709.28), SIMDE_FLOAT64_C(  -506.40) } },
    { { SIMDE_FLOAT64_C(  -343.42), SIMDE_FLOAT64_C(   357.03), SIMDE_FLOAT64_C(   553.95), SIMDE_FLOAT64_C(  -897.12),
        SIMDE_FLOAT64_C(   597.61), SIMDE_FLOAT64_C(   533.05), SIMDE_FLOAT64_C(   856.49), SIMDE_FLOAT64_C(  -226.01) },
      UINT8_C( 32),
      { SIMDE_FLOAT64_C(  -402.06), SIMDE_FLOAT64_C(  -166.69), SIMDE_FLOAT64_C(   744.83), SIMDE_FLOAT64_C(    79.54),
        SIMDE_FLOAT64_C(   945.78), SIMDE_FLOAT64_C(   565.66), SIMDE_FLOAT64_C(   370.26), SIMDE_FLOAT64_C(   136.77) },
      { SIMDE_FLOAT64_C(  -859.22), SIMDE_FLOAT64_C(  -374.15), SIMDE_FLOAT64_C(   247.14), SIMDE_FLOAT64_C(  -937.21) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(  -343.42), SIMDE_FLOAT64_C(   357.03), SIMDE_FLOAT64_C(   553.95), SIMDE_FLOAT64_C(  -897.12),
        SIMDE_FLOAT64_C(   597.61), SIMDE_FLOAT64_C(  -374.15), SIMDE_FLOAT64_C(   856.49), SIMDE_FLOAT64_C(  -226.01) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d src = simde_mm512_loadu_pd(test_vec[i].src);
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m256d b = simde_mm256_loadu_pd(test_vec[i].b);
    simde__m512d r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_insertf64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), test_vec[i].imm8, src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512d src = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m256d b = simde_test_x86_random_f64x4(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m512d r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_insertf64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), imm8, src, k, a, b);

    simde_test_x86_write_f64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_insertf64x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const double a[8];
    const double b[4];
    const int imm8;
    const double r[8];
  } test_vec[] = {
    { UINT8_C(204),
      { SIMDE_FLOAT64_C(  -448.84), SIMDE_FLOAT64_C(    79.07), SIMDE_FLOAT64_C(   803.79), SIMDE_FLOAT64_C(   385.97),
        SIMDE_FLOAT64_C(   448.87), SIMDE_FLOAT64_C(   644.05), SIMDE_FLOAT64_C(  -956.81), SIMDE_FLOAT64_C(   727.91) },
      { SIMDE_FLOAT64_C(    20.17), SIMDE_FLOAT64_C(   291.56), SIMDE_FLOAT64_C(   -87.69), SIMDE_FLOAT64_C(   484.51) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   -87.69), SIMDE_FLOAT64_C(   484.51),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -956.81), SIMDE_FLOAT64_C(   727.91) } },
    { UINT8_C(150),
      { SIMDE_FLOAT64_C(   287.68), SIMDE_FLOAT64_C(   700.68), SIMDE_FLOAT64_C(  -535.84), SIMDE_FLOAT64_C(   338.09),
        SIMDE_FLOAT64_C(  -339.63), SIMDE_FLOAT64_C(   484.59), SIMDE_FLOAT64_C(  -954.07), SIMDE_FLOAT64_C(   109.81) },
      { SIMDE_FLOAT64_C(  -553.94), SIMDE_FLOAT64_C(  -971.55), SIMDE_FLOAT64_C(   -81.93), SIMDE_FLOAT64_C(  -237.60) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   700.68), SIMDE_FLOAT64_C(  -535.84), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(  -553.94), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -237.60) } },
    { UINT8_C(163),
      { SIMDE_FLOAT64_C(   370.50), SIMDE_FLOAT64_C(   647.92), SIMDE_FLOAT64_C(   518.82), SIMDE_FLOAT64_C(   921.66),
        SIMDE_FLOAT64_C(  -273.01), SIMDE_FLOAT64_C(   322.60), SIMDE_FLOAT64_C(   307.64), SIMDE_FLOAT64_C(  -824.15) },
      { SIMDE_FLOAT64_C(   -33.35), SIMDE_FLOAT64_C(   350.83), SIMDE_FLOAT64_C(   903.76), SIMDE_FLOAT64_C(   986.82) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   370.50), SIMDE_FLOAT64_C(   647.92), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   350.83), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   986.82) } },
    { UINT8_C(204),
      { SIMDE_FLOAT64_C(   471.33), SIMDE_FLOAT64_C(   285.14), SIMDE_FLOAT64_C(   612.89), SIMDE_FLOAT64_C(  -240.98),
        SIMDE_FLOAT64_C(   -14.18), SIMDE_FLOAT64_C(  -922.95), SIMDE_FLOAT64_C(  -902.89), SIMDE_FLOAT64_C(   646.19) },
      { SIMDE_FLOAT64_C(   561.65), SIMDE_FLOAT64_C(  -856.96), SIMDE_FLOAT64_C(  -244.01), SIMDE_FLOAT64_C(  -992.30) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   612.89), SIMDE_FLOAT64_C(  -240.98),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -244.01), SIMDE_FLOAT64_C(  -992.30) } },
    { UINT8_C(237),
      { SIMDE_FLOAT64_C(  -229.90), SIMDE_FLOAT64_C(   519.15), SIMDE_FLOAT64_C(   700.82), SIMDE_FLOAT64_C(  -859.39),
        SIMDE_FLOAT64_C(   167.07), SIMDE_FLOAT64_C(   219.64), SIMDE_FLOAT64_C(  -937.73), SIMDE_FLOAT64_C(   894.06) },
      { SIMDE_FLOAT64_C(  -457.76), SIMDE_FLOAT64_C(   369.91), SIMDE_FLOAT64_C(  -930.09), SIMDE_FLOAT64_C(   508.89) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(  -457.76), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -930.09), SIMDE_FLOAT64_C(   508.89),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   219.64), SIMDE_FLOAT64_C(  -937.73), SIMDE_FLOAT64_C(   894.06) } },
    { UINT8_C(129),
      { SIMDE_FLOAT64_C(   495.71), SIMDE_FLOAT64_C(   363.13), SIMDE_FLOAT64_C(  -210.26), SIMDE_FLOAT64_C(   -32.95),
        SIMDE_FLOAT64_C(  -351.73), SIMDE_FLOAT64_C(  -597.36), SIMDE_FLOAT64_C(   726.06), SIMDE_FLOAT64_C(   634.09) },
      { SIMDE_FLOAT64_C(  -520.31), SIMDE_FLOAT64_C(   823.17), SIMDE_FLOAT64_C(   280.28), SIMDE_FLOAT64_C(  -958.66) },
       INT32_C(           1),
      { SIMDE_FLOAT64_C(   495.71), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -958.66) } },
    { UINT8_C(129),
      { SIMDE_FLOAT64_C(  -950.96), SIMDE_FLOAT64_C(  -862.29), SIMDE_FLOAT64_C(   710.34), SIMDE_FLOAT64_C(  -180.86),
        SIMDE_FLOAT64_C(   656.86), SIMDE_FLOAT64_C(   411.16), SIMDE_FLOAT64_C(   -40.25), SIMDE_FLOAT64_C(  -176.07) },
      { SIMDE_FLOAT64_C(  -369.20), SIMDE_FLOAT64_C(    22.03), SIMDE_FLOAT64_C(  -282.01), SIMDE_FLOAT64_C(   173.05) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(  -369.20), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -176.07) } },
    { UINT8_C(  9),
      { SIMDE_FLOAT64_C(  -318.06), SIMDE_FLOAT64_C(   112.67), SIMDE_FLOAT64_C(  -238.42), SIMDE_FLOAT64_C(  -822.34),
        SIMDE_FLOAT64_C(  -524.20), SIMDE_FLOAT64_C(   551.32), SIMDE_FLOAT64_C(   144.70), SIMDE_FLOAT64_C(   124.07) },
      { SIMDE_FLOAT64_C(   953.96), SIMDE_FLOAT64_C(  -129.23), SIMDE_FLOAT64_C(  -241.84), SIMDE_FLOAT64_C(  -566.35) },
       INT32_C(           0),
      { SIMDE_FLOAT64_C(   953.96), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -566.35),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m256d b = simde_mm256_loadu_pd(test_vec[i].b);
    simde__m512d r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_insertf64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), test_vec[i].imm8, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m256d b = simde_test_x86_random_f64x4(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m512d r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_insertf64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), imm8, k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_inserti32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    const int32_t b[4];
    const int imm8;
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  2011462940),  INT32_C(  1964677401),  INT32_C(   601057305),  INT32_C(     1160340),  INT32_C(   159687598), -INT32_C(  1610239808), -INT32_C(  1121293834),  INT32_C(  1694573641),
        -INT32_C(  1143085662), -INT32_C(  1741555842), -INT32_C(   189070241),  INT32_C(  1744096440),  INT32_C(   812677488),  INT32_C(   550532649),  INT32_C(   685636319), -INT32_C(  1047601377) },
      {  INT32_C(  1132227524), -INT32_C(  1042567838),  INT32_C(  1790285490), -INT32_C(   741234077) },
       INT32_C(           3),
      {  INT32_C(  2011462940),  INT32_C(  1964677401),  INT32_C(   601057305),  INT32_C(     1160340),  INT32_C(   159687598), -INT32_C(  1610239808), -INT32_C(  1121293834),  INT32_C(  1694573641),
        -INT32_C(  1143085662), -INT32_C(  1741555842), -INT32_C(   189070241),  INT32_C(  1744096440),  INT32_C(  1132227524), -INT32_C(  1042567838),  INT32_C(  1790285490), -INT32_C(   741234077) } },
    { { -INT32_C(  1202977982), -INT32_C(   828937004),  INT32_C(   703447114), -INT32_C(  1175605682), -INT32_C(   669306837), -INT32_C(  1567958005),  INT32_C(  1007023506),  INT32_C(   140499142),
        -INT32_C(  1346327589),  INT32_C(  1652447256),  INT32_C(  1720412952),  INT32_C(  1159690266), -INT32_C(  1273152855), -INT32_C(  1470715882),  INT32_C(  1675910045),  INT32_C(   258753331) },
      {  INT32_C(   129903855), -INT32_C(  1670824828), -INT32_C(  1039993688),  INT32_C(   352788844) },
       INT32_C(           3),
      { -INT32_C(  1202977982), -INT32_C(   828937004),  INT32_C(   703447114), -INT32_C(  1175605682), -INT32_C(   669306837), -INT32_C(  1567958005),  INT32_C(  1007023506),  INT32_C(   140499142),
        -INT32_C(  1346327589),  INT32_C(  1652447256),  INT32_C(  1720412952),  INT32_C(  1159690266),  INT32_C(   129903855), -INT32_C(  1670824828), -INT32_C(  1039993688),  INT32_C(   352788844) } },
    { { -INT32_C(   864892380),  INT32_C(  2087262752),  INT32_C(  1118817791),  INT32_C(  1697758777), -INT32_C(  1175832451), -INT32_C(  1788770655), -INT32_C(  1442700407),  INT32_C(  1325799210),
         INT32_C(    35354849), -INT32_C(  1853979246), -INT32_C(  1949029038),  INT32_C(  1777337836), -INT32_C(   551364034), -INT32_C(   361397151), -INT32_C(   762021977),  INT32_C(  1881250446) },
      { -INT32_C(  1519240174),  INT32_C(   322367681),  INT32_C(   161352221),  INT32_C(  1316130576) },
       INT32_C(           1),
      { -INT32_C(   864892380),  INT32_C(  2087262752),  INT32_C(  1118817791),  INT32_C(  1697758777), -INT32_C(  1519240174),  INT32_C(   322367681),  INT32_C(   161352221),  INT32_C(  1316130576),
         INT32_C(    35354849), -INT32_C(  1853979246), -INT32_C(  1949029038),  INT32_C(  1777337836), -INT32_C(   551364034), -INT32_C(   361397151), -INT32_C(   762021977),  INT32_C(  1881250446) } },
    { {  INT32_C(   432680341),  INT32_C(   432059554), -INT32_C(   475491767), -INT32_C(   252307277),  INT32_C(  2058460042), -INT32_C(   594033199), -INT32_C(   219373213), -INT32_C(  1470350829),
         INT32_C(   180430439),  INT32_C(   606307035), -INT32_C(   939013356),  INT32_C(  1840840163),  INT32_C(  1793550744), -INT32_C(  1840873682),  INT32_C(   864301600), -INT32_C(   740499348) },
      { -INT32_C(   488792825),  INT32_C(   872808735), -INT32_C(  1325658676), -INT32_C(  1558334453) },
       INT32_C(           1),
      {  INT32_C(   432680341),  INT32_C(   432059554), -INT32_C(   475491767), -INT32_C(   252307277), -INT32_C(   488792825),  INT32_C(   872808735), -INT32_C(  1325658676), -INT32_C(  1558334453),
         INT32_C(   180430439),  INT32_C(   606307035), -INT32_C(   939013356),  INT32_C(  1840840163),  INT32_C(  1793550744), -INT32_C(  1840873682),  INT32_C(   864301600), -INT32_C(   740499348) } },
    { { -INT32_C(  2075390715), -INT32_C(  2052792749),  INT32_C(  1139922786),  INT32_C(  1363854771), -INT32_C(  1552929630),  INT32_C(  1064346674),  INT32_C(  1414144160),  INT32_C(  1114828349),
         INT32_C(  1338425083), -INT32_C(    19633508), -INT32_C(   180238782),  INT32_C(   759597963), -INT32_C(   372131913), -INT32_C(    64405157), -INT32_C(  1638894751),  INT32_C(  1575076449) },
      {  INT32_C(   481077120),  INT32_C(  1411088402), -INT32_C(   783721402), -INT32_C(  1610641176) },
       INT32_C(           3),
      { -INT32_C(  2075390715), -INT32_C(  2052792749),  INT32_C(  1139922786),  INT32_C(  1363854771), -INT32_C(  1552929630),  INT32_C(  1064346674),  INT32_C(  1414144160),  INT32_C(  1114828349),
         INT32_C(  1338425083), -INT32_C(    19633508), -INT32_C(   180238782),  INT32_C(   759597963),  INT32_C(   481077120),  INT32_C(  1411088402), -INT32_C(   783721402), -INT32_C(  1610641176) } },
    { {  INT32_C(   295864784),  INT32_C(   628268722), -INT32_C(  1316548369), -INT32_C(  1724717839),  INT32_C(   279662224), -INT32_C(   967311511), -INT32_C(   659675064), -INT32_C(   148943577),
        -INT32_C(  2012691754),  INT32_C(  1336834656),  INT32_C(  2080454027), -INT32_C(  1458228455), -INT32_C(   340148095),  INT32_C(   145821887),  INT32_C(  1625317177), -INT32_C(  2091384660) },
      {  INT32_C(   571171010),  INT32_C(  1718794715),  INT32_C(   132281326),  INT32_C(   665909414) },
       INT32_C(           0),
      {  INT32_C(   571171010),  INT32_C(  1718794715),  INT32_C(   132281326),  INT32_C(   665909414),  INT32_C(   279662224), -INT32_C(   967311511), -INT32_C(   659675064), -INT32_C(   148943577),
        -INT32_C(  2012691754),  INT32_C(  1336834656),  INT32_C(  2080454027), -INT32_C(  1458228455), -INT32_C(   340148095),  INT32_C(   145821887),  INT32_C(  1625317177), -INT32_C(  2091384660) } },
    { {  INT32_C(  2054689386),  INT32_C(   582189251),  INT32_C(  1624183904), -INT32_C(   870165908),  INT32_C(   396838237),  INT32_C(   704974263), -INT32_C(   389018128),  INT32_C(   664860605),
        -INT32_C(   845015030), -INT32_C(   101689960), -INT32_C(   715538583),  INT32_C(  1856142353),  INT32_C(  2022001089),  INT32_C(  1201834839),  INT32_C(  1429172888),  INT32_C(  1937624937) },
      { -INT32_C(  2143215640), -INT32_C(   562482828),  INT32_C(    28562416),  INT32_C(   275731791) },
       INT32_C(           3),
      {  INT32_C(  2054689386),  INT32_C(   582189251),  INT32_C(  1624183904), -INT32_C(   870165908),  INT32_C(   396838237),  INT32_C(   704974263), -INT32_C(   389018128),  INT32_C(   664860605),
        -INT32_C(   845015030), -INT32_C(   101689960), -INT32_C(   715538583),  INT32_C(  1856142353), -INT32_C(  2143215640), -INT32_C(   562482828),  INT32_C(    28562416),  INT32_C(   275731791) } },
    { { -INT32_C(  2131326731), -INT32_C(  1676133078),  INT32_C(   990211436),  INT32_C(   153319914), -INT32_C(   344021830), -INT32_C(   254059491),  INT32_C(  1698683919),  INT32_C(  1074024267),
         INT32_C(    29424343), -INT32_C(  1549936585),  INT32_C(   819897158), -INT32_C(   700841444), -INT32_C(  1010714458),  INT32_C(   599038996), -INT32_C(  1014434952),  INT32_C(   436505667) },
      { -INT32_C(  1122188154), -INT32_C(   480200291),  INT32_C(  2014527324), -INT32_C(   414298815) },
       INT32_C(           1),
      { -INT32_C(  2131326731), -INT32_C(  1676133078),  INT32_C(   990211436),  INT32_C(   153319914), -INT32_C(  1122188154), -INT32_C(   480200291),  INT32_C(  2014527324), -INT32_C(   414298815),
         INT32_C(    29424343), -INT32_C(  1549936585),  INT32_C(   819897158), -INT32_C(   700841444), -INT32_C(  1010714458),  INT32_C(   599038996), -INT32_C(  1014434952),  INT32_C(   436505667) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi32(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_i32x16(simde_mm512_inserti32x4(a, b, 0), simde_mm512_loadu_epi32(test_vec[i].r)); break;
      case 1: simde_test_x86_assert_equal_i32x16(simde_mm512_inserti32x4(a, b, 1), simde_mm512_loadu_epi32(test_vec[i].r)); break;
      case 2: simde_test_x86_assert_equal_i32x16(simde_mm512_inserti32x4(a, b, 2), simde_mm512_loadu_epi32(test_vec[i].r)); break;
      case 3: simde_test_x86_assert_equal_i32x16(simde_mm512_inserti32x4(a, b, 3), simde_mm512_loadu_epi32(test_vec[i].r)); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    int imm8 = simde_test_codegen_rand() & 3;
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_inserti32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, a, b);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_inserti32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[4];
    const int imm8;
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(   123241083), -INT32_C(   900882945),  INT32_C(  2072004312), -INT32_C(  2059193009),  INT32_C(  1288374372), -INT32_C(  1435939782), -INT32_C(   648900601), -INT32_C(  2005038333),
         INT32_C(   125903880), -INT32_C(  1714303295),  INT32_C(  1662341396), -INT32_C(   337028985), -INT32_C(  1690848159),  INT32_C(   138780928),  INT32_C(   887199537),  INT32_C(   717053474) },
      UINT16_C(16002),
      {  INT32_C(    51135537), -INT32_C(   229367587),  INT32_C(  1816910723), -INT32_C(    31347513), -INT32_C(  1935728313),  INT32_C(   170119465), -INT32_C(  1033353979), -INT32_C(  1610552465),
         INT32_C(   212012079), -INT32_C(  1325402068), -INT32_C(  1709421869),  INT32_C(  1033453046), -INT32_C(  2000078753),  INT32_C(   261352457), -INT32_C(  1596851407),  INT32_C(   356569830) },
      {  INT32_C(   186836190), -INT32_C(  1346690596),  INT32_C(  1632294763),  INT32_C(  1956569877) },
       INT32_C(           3),
      { -INT32_C(   123241083), -INT32_C(   229367587),  INT32_C(  2072004312), -INT32_C(  2059193009),  INT32_C(  1288374372), -INT32_C(  1435939782), -INT32_C(   648900601), -INT32_C(  1610552465),
         INT32_C(   125903880), -INT32_C(  1325402068), -INT32_C(  1709421869),  INT32_C(  1033453046),  INT32_C(   186836190), -INT32_C(  1346690596),  INT32_C(   887199537),  INT32_C(   717053474) } },
    { {  INT32_C(  1394998375), -INT32_C(  1954270065), -INT32_C(   680451067),  INT32_C(  1219921764), -INT32_C(   903560791),  INT32_C(  1396036732),  INT32_C(    23631390), -INT32_C(  1692607436),
         INT32_C(  1760444889),  INT32_C(  2079552117), -INT32_C(    95263338), -INT32_C(  1790768916),  INT32_C(  1163880393),  INT32_C(  1503237179),  INT32_C(  1582956842), -INT32_C(  1208387618) },
      UINT16_C(59320),
      {  INT32_C(   324611615), -INT32_C(    75960407),  INT32_C(   738485481),  INT32_C(  1486147065), -INT32_C(  1393766637), -INT32_C(  2085808599),  INT32_C(  1845136244),  INT32_C(  1649718082),
        -INT32_C(  1971999263), -INT32_C(  2054820452),  INT32_C(  1253214545),  INT32_C(  1789019735),  INT32_C(  1041665557),  INT32_C(   415417252), -INT32_C(  1853506481),  INT32_C(  1358158191) },
      {  INT32_C(   584738950), -INT32_C(  1498980267),  INT32_C(  1089493481), -INT32_C(  1247112545) },
       INT32_C(           0),
      {  INT32_C(  1394998375), -INT32_C(  1954270065), -INT32_C(   680451067), -INT32_C(  1247112545), -INT32_C(  1393766637), -INT32_C(  2085808599),  INT32_C(    23631390),  INT32_C(  1649718082),
        -INT32_C(  1971999263), -INT32_C(  2054820452),  INT32_C(  1253214545), -INT32_C(  1790768916),  INT32_C(  1163880393),  INT32_C(   415417252), -INT32_C(  1853506481),  INT32_C(  1358158191) } },
    { { -INT32_C(  2067467328),  INT32_C(  1926487221),  INT32_C(   987849825), -INT32_C(  1061146024),  INT32_C(  1813438988), -INT32_C(   480854902),  INT32_C(  1065588397),  INT32_C(    23083073),
        -INT32_C(   511368149),  INT32_C(  1649629185),  INT32_C(   362624189),  INT32_C(  1943362918), -INT32_C(   891294912),  INT32_C(  1437414824),  INT32_C(   227815628), -INT32_C(  1810959256) },
      UINT16_C(37913),
      { -INT32_C(   924050827),  INT32_C(   435988860), -INT32_C(  1804115010), -INT32_C(  1249921066),  INT32_C(   787163009),  INT32_C(   291485564), -INT32_C(   771373116), -INT32_C(   798613925),
        -INT32_C(  1248308424), -INT32_C(  1144089092), -INT32_C(   833665544),  INT32_C(  2105790204),  INT32_C(  1923903222), -INT32_C(   377287899),  INT32_C(   784107986), -INT32_C(   520215897) },
      {  INT32_C(  1922406261),  INT32_C(   606954284), -INT32_C(  1510835031),  INT32_C(  1076065610) },
       INT32_C(           0),
      {  INT32_C(  1922406261),  INT32_C(  1926487221),  INT32_C(   987849825),  INT32_C(  1076065610),  INT32_C(   787163009), -INT32_C(   480854902),  INT32_C(  1065588397),  INT32_C(    23083073),
        -INT32_C(   511368149),  INT32_C(  1649629185), -INT32_C(   833665544),  INT32_C(  1943362918),  INT32_C(  1923903222),  INT32_C(  1437414824),  INT32_C(   227815628), -INT32_C(   520215897) } },
    { { -INT32_C(   636898353), -INT32_C(  1079184586), -INT32_C(   781722961),  INT32_C(  1883719641),  INT32_C(  1067235804),  INT32_C(  1659420902),  INT32_C(   665620146), -INT32_C(  2146701903),
        -INT32_C(   698739296), -INT32_C(  1214904824), -INT32_C(  1165427231),  INT32_C(   539676484),  INT32_C(  1851770504),  INT32_C(   953174150), -INT32_C(  2023785002),  INT32_C(   168258410) },
      UINT16_C(24960),
      {  INT32_C(  1986496736), -INT32_C(   948746177),  INT32_C(   748140290),  INT32_C(   938614743),  INT32_C(  1585412238),  INT32_C(   282809776), -INT32_C(   478460452),  INT32_C(   809827151),
        -INT32_C(  1029264509), -INT32_C(   158786828), -INT32_C(  1457315631), -INT32_C(   824175296),  INT32_C(  1043095438), -INT32_C(  1873934156), -INT32_C(  1653356211),  INT32_C(  1204664260) },
      {  INT32_C(  1443459938),  INT32_C(  1598919565), -INT32_C(   217550669),  INT32_C(   331475077) },
       INT32_C(           3),
      { -INT32_C(   636898353), -INT32_C(  1079184586), -INT32_C(   781722961),  INT32_C(  1883719641),  INT32_C(  1067235804),  INT32_C(  1659420902),  INT32_C(   665620146),  INT32_C(   809827151),
        -INT32_C(  1029264509), -INT32_C(  1214904824), -INT32_C(  1165427231),  INT32_C(   539676484),  INT32_C(  1851770504),  INT32_C(  1598919565), -INT32_C(   217550669),  INT32_C(   168258410) } },
    { { -INT32_C(   151301394),  INT32_C(  1782811552), -INT32_C(  1255218946),  INT32_C(   555185837),  INT32_C(   313421439),  INT32_C(   734399931), -INT32_C(    38749931),  INT32_C(  1766114427),
        -INT32_C(  1235271914), -INT32_C(   937385270),  INT32_C(   813518723),  INT32_C(  1146197445), -INT32_C(  1101594621),  INT32_C(   602479885),  INT32_C(  1361091030),  INT32_C(  1941595485) },
      UINT16_C( 6564),
      {  INT32_C(  1253863210), -INT32_C(  1248248009), -INT32_C(  1052090769), -INT32_C(    87994973), -INT32_C(   199766517),  INT32_C(   311291377), -INT32_C(   109581505), -INT32_C(  2012013730),
        -INT32_C(  1026306166),  INT32_C(  2104978446),  INT32_C(  1849606603),  INT32_C(   426311438), -INT32_C(  1089634354), -INT32_C(  1412326804), -INT32_C(   458930043), -INT32_C(   294864796) },
      { -INT32_C(  1783611513),  INT32_C(  1997678508), -INT32_C(   152743448),  INT32_C(   504319312) },
       INT32_C(           0),
      { -INT32_C(   151301394),  INT32_C(  1782811552), -INT32_C(   152743448),  INT32_C(   555185837),  INT32_C(   313421439),  INT32_C(   311291377), -INT32_C(    38749931), -INT32_C(  2012013730),
        -INT32_C(  1026306166), -INT32_C(   937385270),  INT32_C(   813518723),  INT32_C(   426311438), -INT32_C(  1089634354),  INT32_C(   602479885),  INT32_C(  1361091030),  INT32_C(  1941595485) } },
    { { -INT32_C(  1221009892), -INT32_C(   130227025),  INT32_C(  1096556681), -INT32_C(   859288947),  INT32_C(   578313723), -INT32_C(  1072959633),  INT32_C(   554762708),  INT32_C(   770518801),
        -INT32_C(  1109121523), -INT32_C(  1816846070), -INT32_C(   824962751),  INT32_C(  1453038683),  INT32_C(  1752765433), -INT32_C(   685145085), -INT32_C(  1762051451),  INT32_C(  2009392745) },
      UINT16_C(43020),
      { -INT32_C(   372697548),  INT32_C(  2113538985),  INT32_C(  1947817433),  INT32_C(   629674668), -INT32_C(  1532392581),  INT32_C(  1541287522), -INT32_C(  1992144955), -INT32_C(   231649602),
         INT32_C(   249297765), -INT32_C(   544418554), -INT32_C(   682318549),  INT32_C(   872209592), -INT32_C(   908614297), -INT32_C(  1742359085), -INT32_C(  1155438596),  INT32_C(   447566773) },
      {  INT32_C(  1412008270), -INT32_C(  1993099938),  INT32_C(   308315994), -INT32_C(   901424029) },
       INT32_C(           1),
      { -INT32_C(  1221009892), -INT32_C(   130227025),  INT32_C(  1947817433),  INT32_C(   629674668),  INT32_C(   578313723), -INT32_C(  1072959633),  INT32_C(   554762708),  INT32_C(   770518801),
        -INT32_C(  1109121523), -INT32_C(  1816846070), -INT32_C(   824962751),  INT32_C(   872209592),  INT32_C(  1752765433), -INT32_C(  1742359085), -INT32_C(  1762051451),  INT32_C(   447566773) } },
    { { -INT32_C(   774532068),  INT32_C(   550399417), -INT32_C(   489322097), -INT32_C(  1087311818), -INT32_C(   836860903),  INT32_C(  1059628983),  INT32_C(  1688353544), -INT32_C(  1654231680),
        -INT32_C(  1167181055),  INT32_C(   937049256), -INT32_C(    65425467), -INT32_C(  1195685473), -INT32_C(  2054759987), -INT32_C(  1983598719),  INT32_C(  1793943274), -INT32_C(   737717293) },
      UINT16_C(30350),
      {  INT32_C(  1756510094), -INT32_C(  2011662226),  INT32_C(   802273140), -INT32_C(   167141520), -INT32_C(   391804380),  INT32_C(     5148435),  INT32_C(    22291194), -INT32_C(  2055740682),
        -INT32_C(  1997723111),  INT32_C(   370148770),  INT32_C(   759554493), -INT32_C(  1541190016), -INT32_C(   343094824),  INT32_C(  1391254360), -INT32_C(   212582147),  INT32_C(  1014549282) },
      { -INT32_C(  1748736267),  INT32_C(   682480747),  INT32_C(   911602613),  INT32_C(   450525505) },
       INT32_C(           2),
      { -INT32_C(   774532068), -INT32_C(  2011662226),  INT32_C(   802273140), -INT32_C(   167141520), -INT32_C(   836860903),  INT32_C(  1059628983),  INT32_C(  1688353544), -INT32_C(  2055740682),
        -INT32_C(  1167181055),  INT32_C(   682480747),  INT32_C(   911602613), -INT32_C(  1195685473), -INT32_C(   343094824),  INT32_C(  1391254360), -INT32_C(   212582147), -INT32_C(   737717293) } },
    { {  INT32_C(  1117390183),  INT32_C(   826273265),  INT32_C(   206844481),  INT32_C(   285380779),  INT32_C(   679254356),  INT32_C(   987604295),  INT32_C(  1937445882),  INT32_C(  1437963758),
        -INT32_C(  1936240485),  INT32_C(  2126435901), -INT32_C(  1282797048), -INT32_C(   154891102),  INT32_C(  1830699302), -INT32_C(   525861914), -INT32_C(    44817905),  INT32_C(  1397885368) },
      UINT16_C(59737),
      { -INT32_C(  1648388385), -INT32_C(  1615870188),  INT32_C(  1059803515),  INT32_C(  1702908231),  INT32_C(  1700882110), -INT32_C(  1685557177), -INT32_C(  1079754643),  INT32_C(  1923677843),
        -INT32_C(  1458542700), -INT32_C(  1454850258),  INT32_C(  1491694353), -INT32_C(  2084673083),  INT32_C(   401154000), -INT32_C(    55414385),  INT32_C(  1153128113), -INT32_C(   373922988) },
      { -INT32_C(   124598582), -INT32_C(  1750934906),  INT32_C(   317688653), -INT32_C(   996758028) },
       INT32_C(           0),
      { -INT32_C(   124598582),  INT32_C(   826273265),  INT32_C(   206844481), -INT32_C(   996758028),  INT32_C(  1700882110),  INT32_C(   987604295), -INT32_C(  1079754643),  INT32_C(  1437963758),
        -INT32_C(  1458542700),  INT32_C(  2126435901), -INT32_C(  1282797048), -INT32_C(  2084673083),  INT32_C(  1830699302), -INT32_C(    55414385),  INT32_C(  1153128113), -INT32_C(   373922988) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_inserti32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), test_vec[i].imm8, src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    int imm8 = simde_test_codegen_rand() & 3;
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_inserti32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, src, k, a, b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_inserti32x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[4];
    const int imm8;
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(24510),
      {  INT32_C(   948879766),  INT32_C(  1992127389),  INT32_C(   682239486),  INT32_C(  1412557776), -INT32_C(  1318727619), -INT32_C(   330215499),  INT32_C(  1533645226), -INT32_C(  1497749489),
        -INT32_C(  2032252695), -INT32_C(  1157784644), -INT32_C(  1864194112), -INT32_C(   790359150), -INT32_C(  1719580188),  INT32_C(  1116066456),  INT32_C(  1922953059),  INT32_C(     1595159) },
      {  INT32_C(  1552414368),  INT32_C(  1360430225), -INT32_C(  1092421589), -INT32_C(   275855861) },
       INT32_C(           0),
      {  INT32_C(           0),  INT32_C(  1360430225), -INT32_C(  1092421589), -INT32_C(   275855861), -INT32_C(  1318727619), -INT32_C(   330215499),  INT32_C(           0), -INT32_C(  1497749489),
        -INT32_C(  2032252695), -INT32_C(  1157784644), -INT32_C(  1864194112), -INT32_C(   790359150), -INT32_C(  1719580188),  INT32_C(           0),  INT32_C(  1922953059),  INT32_C(           0) } },
    { UINT16_C(35087),
      { -INT32_C(   351346264), -INT32_C(  1232536252),  INT32_C(   365944852), -INT32_C(   593705600), -INT32_C(  1460527017),  INT32_C(   160098891), -INT32_C(   443067915),  INT32_C(   158246497),
        -INT32_C(   873169785), -INT32_C(  1904050822), -INT32_C(   576499363),  INT32_C(  1840856854), -INT32_C(  1424577697), -INT32_C(  1967873899),  INT32_C(  1383025905),  INT32_C(  2052906482) },
      { -INT32_C(   733654950),  INT32_C(   711116749),  INT32_C(   788989464), -INT32_C(  1516453819) },
       INT32_C(           3),
      { -INT32_C(   351346264), -INT32_C(  1232536252),  INT32_C(   365944852), -INT32_C(   593705600),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
        -INT32_C(   873169785),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1840856854),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1516453819) } },
    { UINT16_C(20658),
      { -INT32_C(  1962650880), -INT32_C(  1745203132), -INT32_C(  1108092861),  INT32_C(   100811570),  INT32_C(   996723217),  INT32_C(   289631970),  INT32_C(  1504576436),  INT32_C(  1873371247),
        -INT32_C(   117789261),  INT32_C(  1099953662), -INT32_C(       97587), -INT32_C(   704315195), -INT32_C(  1391301174), -INT32_C(  1866574372), -INT32_C(   941003688), -INT32_C(  2143841587) },
      {  INT32_C(  1031287103), -INT32_C(   209844442),  INT32_C(  1341291657),  INT32_C(  1193670525) },
       INT32_C(           1),
      {  INT32_C(           0), -INT32_C(  1745203132),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1031287103), -INT32_C(   209844442),  INT32_C(           0),  INT32_C(  1193670525),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1391301174),  INT32_C(           0), -INT32_C(   941003688),  INT32_C(           0) } },
    { UINT16_C(62519),
      { -INT32_C(   776762303), -INT32_C(  1397088283),  INT32_C(  1843612908), -INT32_C(   907733621),  INT32_C(   793242939), -INT32_C(   987642762), -INT32_C(  1997858495), -INT32_C(  1082318210),
        -INT32_C(  1802489681),  INT32_C(  1010846287),  INT32_C(   564733078),  INT32_C(  1961528889), -INT32_C(   224185733),  INT32_C(   934790646),  INT32_C(  1556062942),  INT32_C(  1947942085) },
      { -INT32_C(  1140282516), -INT32_C(  1946662667), -INT32_C(  1515413140), -INT32_C(  1424320721) },
       INT32_C(           1),
      { -INT32_C(   776762303), -INT32_C(  1397088283),  INT32_C(  1843612908),  INT32_C(           0), -INT32_C(  1140282516), -INT32_C(  1946662667),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(   564733078),  INT32_C(           0), -INT32_C(   224185733),  INT32_C(   934790646),  INT32_C(  1556062942),  INT32_C(  1947942085) } },
    { UINT16_C(40381),
      { -INT32_C(   162233665), -INT32_C(  1128925344),  INT32_C(   819458748),  INT32_C(   439911006),  INT32_C(    34766967), -INT32_C(  1833978899), -INT32_C(  1901312285), -INT32_C(   836015601),
         INT32_C(  1304723436),  INT32_C(   839481718), -INT32_C(   899489429), -INT32_C(   605775261),  INT32_C(   148764443), -INT32_C(  1919251286), -INT32_C(   518305838), -INT32_C(  1649457487) },
      {  INT32_C(  1021998022),  INT32_C(  1483732205),  INT32_C(   941806037), -INT32_C(  2028796052) },
       INT32_C(           2),
      { -INT32_C(   162233665),  INT32_C(           0),  INT32_C(   819458748),  INT32_C(   439911006),  INT32_C(    34766967), -INT32_C(  1833978899),  INT32_C(           0), -INT32_C(   836015601),
         INT32_C(  1021998022),  INT32_C(           0),  INT32_C(   941806037), -INT32_C(  2028796052),  INT32_C(   148764443),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1649457487) } },
    { UINT16_C(36849),
      {  INT32_C(   908688808),  INT32_C(   827420752), -INT32_C(  1075734495), -INT32_C(  1700178850), -INT32_C(  1710645951), -INT32_C(  1413686414), -INT32_C(   826358969),  INT32_C(  1784524737),
         INT32_C(  2107672109),  INT32_C(   414118391), -INT32_C(   405303415),  INT32_C(   629309924), -INT32_C(  1849717986), -INT32_C(  1388544922),  INT32_C(    24902207), -INT32_C(   680798038) },
      {  INT32_C(  1448348511), -INT32_C(  2039610628),  INT32_C(  1986872978), -INT32_C(   442765369) },
       INT32_C(           3),
      {  INT32_C(   908688808),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1710645951), -INT32_C(  1413686414), -INT32_C(   826358969),  INT32_C(  1784524737),
         INT32_C(  2107672109),  INT32_C(   414118391), -INT32_C(   405303415),  INT32_C(   629309924),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   442765369) } },
    { UINT16_C(30299),
      { -INT32_C(  1884104735),  INT32_C(   386575638),  INT32_C(   780329815), -INT32_C(  1736274622),  INT32_C(   268862602), -INT32_C(  1937945578), -INT32_C(   114791149), -INT32_C(   915438616),
         INT32_C(  1884824154),  INT32_C(   646472655), -INT32_C(  2007758266),  INT32_C(   572577432),  INT32_C(  1899112026), -INT32_C(  2030194829),  INT32_C(    75441436),  INT32_C(    47116200) },
      { -INT32_C(   529324527), -INT32_C(   821625975), -INT32_C(  1655219707), -INT32_C(  1967163600) },
       INT32_C(           2),
      { -INT32_C(  1884104735),  INT32_C(   386575638),  INT32_C(           0), -INT32_C(  1736274622),  INT32_C(   268862602),  INT32_C(           0), -INT32_C(   114791149),  INT32_C(           0),
         INT32_C(           0), -INT32_C(   821625975), -INT32_C(  1655219707),  INT32_C(           0),  INT32_C(  1899112026), -INT32_C(  2030194829),  INT32_C(    75441436),  INT32_C(           0) } },
    { UINT16_C(64497),
      { -INT32_C(  1745247983), -INT32_C(  1038672195), -INT32_C(   913307961), -INT32_C(   147016169),  INT32_C(   268252992), -INT32_C(   630761667),  INT32_C(   295362183), -INT32_C(  1928492164),
        -INT32_C(   366672340), -INT32_C(   341033948),  INT32_C(  1505049666),  INT32_C(   844165618),  INT32_C(  1715621161),  INT32_C(   725658020),  INT32_C(    71096967), -INT32_C(  1835972250) },
      {  INT32_C(  1954330191),  INT32_C(   878651634),  INT32_C(  1468863588),  INT32_C(   797564166) },
       INT32_C(           2),
      { -INT32_C(  1745247983),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   268252992), -INT32_C(   630761667),  INT32_C(   295362183), -INT32_C(  1928492164),
         INT32_C(  1954330191),  INT32_C(   878651634),  INT32_C(           0),  INT32_C(   797564166),  INT32_C(  1715621161),  INT32_C(   725658020),  INT32_C(    71096967), -INT32_C(  1835972250) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_inserti32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), test_vec[i].imm8, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    int imm8 = simde_test_codegen_rand() & 3;
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_inserti32x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_inserti64x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[8];
    const int64_t b[4];
    const int imm8;
    const int64_t r[8];
  } test_vec[] = {
    { { -INT64_C( 5693401133720103477),  INT64_C( 5246597433765789750), -INT64_C( 5209979076043193563),  INT64_C( 6774799854951270908),
         INT64_C(  146031414961427329), -INT64_C( 1537889364577477426),  INT64_C( 5687458439450059412),  INT64_C( 1721268507576412914) },
      {  INT64_C( 5966834670026798591),  INT64_C( 3837166531510423697),  INT64_C( 6042803075560535869),  INT64_C( 3509072881400677380) },
       INT32_C(           1),
      { -INT64_C( 5693401133720103477),  INT64_C( 5246597433765789750), -INT64_C( 5209979076043193563),  INT64_C( 6774799854951270908),
         INT64_C( 5966834670026798591),  INT64_C( 3837166531510423697),  INT64_C( 6042803075560535869),  INT64_C( 3509072881400677380) } },
    { {  INT64_C( 1956048552865430560),  INT64_C( 3716018053951880186),  INT64_C(  125229962274021470),  INT64_C( 2948839644586989248),
         INT64_C( 2541686231159136607),  INT64_C( 6002183351676895838), -INT64_C( 8635995573007425495),  INT64_C( 1895998654733572126) },
      {  INT64_C( 5484691580701117649), -INT64_C( 4047413771253672564), -INT64_C( 4187481619001712493), -INT64_C( 6322988085866076565) },
       INT32_C(           1),
      {  INT64_C( 1956048552865430560),  INT64_C( 3716018053951880186),  INT64_C(  125229962274021470),  INT64_C( 2948839644586989248),
         INT64_C( 5484691580701117649), -INT64_C( 4047413771253672564), -INT64_C( 4187481619001712493), -INT64_C( 6322988085866076565) } },
    { {  INT64_C( 6168386401021043332), -INT64_C( 3517950195646915608),  INT64_C( 5919203718619170982),  INT64_C( 1201001552025242408),
         INT64_C( 5370745287543283827), -INT64_C( 2625015610674480173),  INT64_C( 7951309952043518362), -INT64_C( 4277720006267863035) },
      {  INT64_C( 5478769107567886428), -INT64_C( 7418938219249537542),  INT64_C( 6872764997045297067),  INT64_C(  832482785543336568) },
       INT32_C(           1),
      {  INT64_C( 6168386401021043332), -INT64_C( 3517950195646915608),  INT64_C( 5919203718619170982),  INT64_C( 1201001552025242408),
         INT64_C( 5478769107567886428), -INT64_C( 7418938219249537542),  INT64_C( 6872764997045297067),  INT64_C(  832482785543336568) } },
    { { -INT64_C( 4612503153071568508), -INT64_C( 2993390484476515288), -INT64_C(  475535721295288329),  INT64_C( 4640030036918218623),
         INT64_C( 8146237984572911740),  INT64_C( 1835691985944169324), -INT64_C( 5271623652586510313), -INT64_C( 5937103093269823586) },
      { -INT64_C( 8424537120928178596),  INT64_C( 1805845606760862163),  INT64_C( 2611471535658506064), -INT64_C( 8273929244304866226) },
       INT32_C(           0),
      { -INT64_C( 8424537120928178596),  INT64_C( 1805845606760862163),  INT64_C( 2611471535658506064), -INT64_C( 8273929244304866226),
         INT64_C( 8146237984572911740),  INT64_C( 1835691985944169324), -INT64_C( 5271623652586510313), -INT64_C( 5937103093269823586) } },
    { { -INT64_C( 7348288781745180119),  INT64_C( 2506453348721163217), -INT64_C( 7940747139185356882), -INT64_C( 6747431858419446710),
        -INT64_C( 1845111259694599295),  INT64_C( 7277860668493988308), -INT64_C( 4641893502607513481),  INT64_C( 2672463773190805242) },
      { -INT64_C( 7498963132896777486), -INT64_C( 1481272561630818987), -INT64_C(  265005380966329569), -INT64_C( 1827467539619220007) },
       INT32_C(           1),
      { -INT64_C( 7348288781745180119),  INT64_C( 2506453348721163217), -INT64_C( 7940747139185356882), -INT64_C( 6747431858419446710),
        -INT64_C( 7498963132896777486), -INT64_C( 1481272561630818987), -INT64_C(  265005380966329569), -INT64_C( 1827467539619220007) } },
    { {  INT64_C( 2971475103301627539),  INT64_C(  925641154200684610),  INT64_C( 3734904380857412472), -INT64_C( 8542186143454047365),
        -INT64_C( 5187675673611687428), -INT64_C( 6865094242915475468),  INT64_C( 5739398828593498292), -INT64_C( 2791169792230175037) },
      {  INT64_C( 6169002818325820526),  INT64_C(  728101369373920594), -INT64_C( 1610279694161300091),  INT64_C( 3289571384520604145) },
       INT32_C(           1),
      {  INT64_C( 2971475103301627539),  INT64_C(  925641154200684610),  INT64_C( 3734904380857412472), -INT64_C( 8542186143454047365),
         INT64_C( 6169002818325820526),  INT64_C(  728101369373920594), -INT64_C( 1610279694161300091),  INT64_C( 3289571384520604145) } },
    { { -INT64_C( 5017765705660296872), -INT64_C( 6443495544218134701), -INT64_C( 8238290343438783081),  INT64_C( 8636629885542758745),
         INT64_C( 8859107301542516618), -INT64_C( 5526446588613230154),  INT64_C( 3039113707380349451),  INT64_C( 8946760142920776082) },
      {  INT64_C( 4737686251077085981),  INT64_C( 8319335330434599333), -INT64_C( 8337217367111900847),  INT64_C( 1813094659912545978) },
       INT32_C(           0),
      {  INT64_C( 4737686251077085981),  INT64_C( 8319335330434599333), -INT64_C( 8337217367111900847),  INT64_C( 1813094659912545978),
         INT64_C( 8859107301542516618), -INT64_C( 5526446588613230154),  INT64_C( 3039113707380349451),  INT64_C( 8946760142920776082) } },
    { {  INT64_C( 7360815958805274332),  INT64_C( 4418101998328342472),  INT64_C( 8452909873430491731),  INT64_C(   34774406344604162),
        -INT64_C( 4581102037239784983), -INT64_C( 3663942751243781745),  INT64_C( 2717270516126801794), -INT64_C( 7220103964047562838) },
      { -INT64_C( 8801707171082318302),  INT64_C( 3138612745657620806),  INT64_C( 5277692780534820854), -INT64_C(  534269373098570680) },
       INT32_C(           0),
      { -INT64_C( 8801707171082318302),  INT64_C( 3138612745657620806),  INT64_C( 5277692780534820854), -INT64_C(  534269373098570680),
        -INT64_C( 4581102037239784983), -INT64_C( 3663942751243781745),  INT64_C( 2717270516126801794), -INT64_C( 7220103964047562838) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi64(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_i64x8(simde_mm512_inserti64x4(a, b, 0), simde_mm512_loadu_epi64(test_vec[i].r)); break;
      case 1: simde_test_x86_assert_equal_i64x8(simde_mm512_inserti64x4(a, b, 1), simde_mm512_loadu_epi64(test_vec[i].r)); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m256i b = simde_test_x86_random_i64x4();
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_inserti64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, a, b);

    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_inserti64x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[4];
    const int imm8;
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 8128429096002904905), -INT64_C( 5324898230582224583),  INT64_C( 9054841775217890047),  INT64_C( 4073881825826770222),
        -INT64_C( 1019124441079432135),  INT64_C( 1057114523657758128),  INT64_C( 6236759978313770354),  INT64_C( 8043583099784913017) },
      UINT8_C( 72),
      { -INT64_C( 7335535081177425366), -INT64_C( 5149101106665019529),  INT64_C(  675318018428935034), -INT64_C( 7922202933783967943),
         INT64_C( 8635446677833191048), -INT64_C( 5622715310163394451),  INT64_C( 5325214157414848949),  INT64_C( 6608782671374853565) },
      {  INT64_C( 9052921415721339435),  INT64_C(  133337002689770224),  INT64_C( 2731368747861176718),  INT64_C( 4402551290744993258) },
       INT32_C(           1),
      {  INT64_C( 8128429096002904905), -INT64_C( 5324898230582224583),  INT64_C( 9054841775217890047), -INT64_C( 7922202933783967943),
        -INT64_C( 1019124441079432135),  INT64_C( 1057114523657758128),  INT64_C( 2731368747861176718),  INT64_C( 8043583099784913017) } },
    { {  INT64_C( 4328618992414189746), -INT64_C( 8837882071330833899), -INT64_C( 3164040567253567640), -INT64_C( 6834172923571940005),
         INT64_C( 8201852465021387659),  INT64_C( 5524112324551268880),  INT64_C(  162288635339183501), -INT64_C( 3101142544363346330) },
      UINT8_C(138),
      { -INT64_C( 7593534908876297799),  INT64_C(  124067153329177236), -INT64_C( 8487690572752477121),  INT64_C( 4115066855160515081),
        -INT64_C( 1233310814219796208),  INT64_C( 7031755240140987433), -INT64_C( 1017137417734308364),  INT64_C( 7902301516541900272) },
      {  INT64_C(  209025793202418334), -INT64_C( 4361124021155725577),  INT64_C(   91195763511306857),  INT64_C(   35215579597161908) },
       INT32_C(           1),
      {  INT64_C( 4328618992414189746),  INT64_C(  124067153329177236), -INT64_C( 3164040567253567640),  INT64_C( 4115066855160515081),
         INT64_C( 8201852465021387659),  INT64_C( 5524112324551268880),  INT64_C(  162288635339183501),  INT64_C(   35215579597161908) } },
    { { -INT64_C( 2465603290756745162),  INT64_C(  474689248116400384), -INT64_C( 1207502178615892085),  INT64_C(  159124400758673033),
         INT64_C(  506263562706263370), -INT64_C( 5917789138614701442), -INT64_C( 7508779251320490319), -INT64_C( 5074868006623044157) },
      UINT8_C( 15),
      { -INT64_C( 7313812912545013146),  INT64_C( 3444524984007678545), -INT64_C( 7937360748202841696),  INT64_C( 6958829959343269905),
         INT64_C( 1666300824636270748), -INT64_C( 1970607117422109891), -INT64_C( 3986016936252546136),  INT64_C(  402189710661496634) },
      {  INT64_C( 1612757991810594437),  INT64_C( 4372488998074815621), -INT64_C( 5434429151340025834),  INT64_C( 4766352932367556790) },
       INT32_C(           0),
      {  INT64_C( 1612757991810594437),  INT64_C( 4372488998074815621), -INT64_C( 5434429151340025834),  INT64_C( 4766352932367556790),
         INT64_C(  506263562706263370), -INT64_C( 5917789138614701442), -INT64_C( 7508779251320490319), -INT64_C( 5074868006623044157) } },
    { { -INT64_C( 9016379153132516177), -INT64_C(   56205560666281489),  INT64_C( 8820350012647605674),  INT64_C( 6630558337768851477),
        -INT64_C( 6740035196471062522), -INT64_C( 8676749978917106882),  INT64_C( 5843317422756007426), -INT64_C( 3588269628282679510) },
      UINT8_C( 55),
      { -INT64_C( 4270385271890394391), -INT64_C(  770352037352091056),  INT64_C( 5993872207350197346),  INT64_C( 2487370421748041062),
         INT64_C( 5307876963713576364), -INT64_C( 3131156835590583633), -INT64_C( 6473636113816175150), -INT64_C( 2637289491169852405) },
      {  INT64_C( 1307848619440735972),  INT64_C( 8429107523598466227), -INT64_C( 1956968136397106447),  INT64_C( 2720612063593941687) },
       INT32_C(           1),
      { -INT64_C( 4270385271890394391), -INT64_C(  770352037352091056),  INT64_C( 5993872207350197346),  INT64_C( 6630558337768851477),
         INT64_C( 1307848619440735972),  INT64_C( 8429107523598466227),  INT64_C( 5843317422756007426), -INT64_C( 3588269628282679510) } },
    { { -INT64_C( 1829293598978866303), -INT64_C( 8081431643047804237), -INT64_C( 6165343361803220666), -INT64_C( 7451186792936810891),
         INT64_C( 5702436309088506132), -INT64_C( 1356410700180247582),  INT64_C( 6315198841098769689), -INT64_C( 5467946501800292499) },
      UINT8_C(132),
      { -INT64_C( 4494188813947440480),  INT64_C(  274533131504599206), -INT64_C(  251853669268076292), -INT64_C( 4391764848541956458),
         INT64_C( 4834935933172395331), -INT64_C( 3110359134051401116), -INT64_C( 8288477129655928175),  INT64_C( 3050665275085461103) },
      {  INT64_C( 7075609104017971537), -INT64_C( 8878272771995194594), -INT64_C( 8483238964075777341), -INT64_C( 3449067371151664367) },
       INT32_C(           0),
      { -INT64_C( 1829293598978866303), -INT64_C( 8081431643047804237), -INT64_C( 8483238964075777341), -INT64_C( 7451186792936810891),
         INT64_C( 5702436309088506132), -INT64_C( 1356410700180247582),  INT64_C( 6315198841098769689),  INT64_C( 3050665275085461103) } },
    { {  INT64_C( 1900426611854524069), -INT64_C( 3266356438992267809),  INT64_C(  405738115411305492),  INT64_C( 7013323686296197781),
         INT64_C( 2890186473953467666), -INT64_C( 3366233199543768490),  INT64_C( 4814352046635431201), -INT64_C( 4847106123749501539) },
      UINT8_C( 60),
      {  INT64_C(  411426094391344733), -INT64_C( 6224938460726023234), -INT64_C( 8316572671201089547),  INT64_C(  558150213688553187),
        -INT64_C( 5796383475474480208), -INT64_C( 2710103860773760697),  INT64_C( 3959059155306854957), -INT64_C( 3961348968189253803) },
      { -INT64_C( 3577578257151464263),  INT64_C( 8571454314521396020), -INT64_C( 9065522655865944666),  INT64_C( 1789203794257635570) },
       INT32_C(           1),
      {  INT64_C( 1900426611854524069), -INT64_C( 3266356438992267809), -INT64_C( 8316572671201089547),  INT64_C(  558150213688553187),
        -INT64_C( 3577578257151464263),  INT64_C( 8571454314521396020),  INT64_C( 4814352046635431201), -INT64_C( 4847106123749501539) } },
    { { -INT64_C( 7043830911678627288),  INT64_C( 1325568249532209618),  INT64_C( 8644308250956553784), -INT64_C(   91216761171082943),
         INT64_C( 5866862653301405361), -INT64_C( 1729253953653222751),  INT64_C( 4672733100372606675),  INT64_C( 5672218437389758179) },
      UINT8_C(250),
      { -INT64_C( 2834230342754690368), -INT64_C( 3017006475906606886),  INT64_C(  729607720769258331),  INT64_C( 5827364970507524383),
        -INT64_C(  257547256132511985), -INT64_C( 4609033561745721958),  INT64_C( 1392771119831703279),  INT64_C(   60470874049346550) },
      { -INT64_C( 3726163330900611284), -INT64_C( 1145032502822398473),  INT64_C( 3381797840249035494), -INT64_C( 1838722389795084918) },
       INT32_C(           0),
      { -INT64_C( 7043830911678627288), -INT64_C( 1145032502822398473),  INT64_C( 8644308250956553784), -INT64_C( 1838722389795084918),
        -INT64_C(  257547256132511985), -INT64_C( 4609033561745721958),  INT64_C( 1392771119831703279),  INT64_C(   60470874049346550) } },
    { {  INT64_C( 1443886911955088656), -INT64_C( 7231930521845668426),  INT64_C( 5861717956854251072),  INT64_C( 6916105500460045493),
         INT64_C( 6033269468337137190),  INT64_C( 1584339997477163231), -INT64_C( 8723684887453446350),  INT64_C( 1351574584201509394) },
      UINT8_C(167),
      {  INT64_C( 4867714394852997171), -INT64_C( 5003447601889481047),  INT64_C( 1757266001990921746), -INT64_C( 5098330386316605993),
        -INT64_C( 4285751072457784369), -INT64_C( 2412043957742428446), -INT64_C(  569826153143004704),  INT64_C(  186845726931688279) },
      { -INT64_C( 3155955771275674029), -INT64_C( 1796330389466229400),  INT64_C( 3049778981871436088),  INT64_C(  382137131508228632) },
       INT32_C(           1),
      {  INT64_C( 4867714394852997171), -INT64_C( 5003447601889481047),  INT64_C( 1757266001990921746),  INT64_C( 6916105500460045493),
         INT64_C( 6033269468337137190), -INT64_C( 1796330389466229400), -INT64_C( 8723684887453446350),  INT64_C(  382137131508228632) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi64(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_inserti64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), test_vec[i].imm8, src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i64x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m256i b = simde_test_x86_random_i64x4();
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_inserti64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, src, k, a, b);

    simde_test_x86_write_i64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_inserti64x4 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[4];
    const int imm8;
    const int64_t r[8];
  } test_vec[] = {
    { UINT8_C(145),
      {  INT64_C(  287037834028355443), -INT64_C( 3437545133234846985),  INT64_C( 5904471213426417596),  INT64_C( 1949932914186691120),
         INT64_C( 2703212208761956892), -INT64_C( 3712801106302095937),  INT64_C( 6644439119405139610), -INT64_C( 3517335828594809491) },
      {  INT64_C( 7901515718180467027), -INT64_C( 1487436571948362113),  INT64_C( 7881022271032079831),  INT64_C( 8551404555756189993) },
       INT32_C(           0),
      {  INT64_C( 7901515718180467027),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C( 2703212208761956892),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 3517335828594809491) } },
    { UINT8_C(251),
      { -INT64_C( 3137517188096896777), -INT64_C( 9120566934224124196),  INT64_C( 6618231906406591807),  INT64_C( 7508255325763205771),
         INT64_C( 2850914129332862974),  INT64_C( 8103607619448468802),  INT64_C( 3706154252432833081),  INT64_C( 2934829808475601889) },
      {  INT64_C( 9083325303390257307),  INT64_C( 8218197654651432298), -INT64_C( 2036203164767282207),  INT64_C( 6899778256777848687) },
       INT32_C(           0),
      {  INT64_C( 9083325303390257307),  INT64_C( 8218197654651432298),  INT64_C(                   0),  INT64_C( 6899778256777848687),
         INT64_C( 2850914129332862974),  INT64_C( 8103607619448468802),  INT64_C( 3706154252432833081),  INT64_C( 2934829808475601889) } },
    { UINT8_C(164),
      { -INT64_C( 2539322747166004154), -INT64_C(  845088561991255010),  INT64_C( 6922691423358175021),  INT64_C( 8958785789338113775),
         INT64_C( 1095100319862451692), -INT64_C( 4111083210463906417),  INT64_C( 2329138576579746502), -INT64_C( 8193695190862486887) },
      {  INT64_C( 4508691120039140242),  INT64_C( 1254866344294009611),  INT64_C( 8600279751833295624), -INT64_C( 2150873100491281674) },
       INT32_C(           0),
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 8600279751833295624),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 4111083210463906417),  INT64_C(                   0), -INT64_C( 8193695190862486887) } },
    { UINT8_C( 10),
      { -INT64_C( 2971764863835022491),  INT64_C( 2329782805707623887), -INT64_C( 1023606105421040570),  INT64_C( 7004556456024480260),
         INT64_C( 5834443558699423113),  INT64_C( 5793564305396530472),  INT64_C( 5454347640071379732), -INT64_C( 1570443032909237290) },
      { -INT64_C( 3399930022410285824),  INT64_C( 5790639818234289154), -INT64_C( 2239766717107581608),  INT64_C(  738281420492070899) },
       INT32_C(           0),
      {  INT64_C(                   0),  INT64_C( 5790639818234289154),  INT64_C(                   0),  INT64_C(  738281420492070899),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 75),
      {  INT64_C(  195997758905130123),  INT64_C( 6055326389186327646), -INT64_C( 6575527098145566346),  INT64_C( 6838153993056471388),
        -INT64_C( 1869992717149587939), -INT64_C( 2882118619760653464),  INT64_C( 8741758583524824506), -INT64_C(  435728913640661823) },
      { -INT64_C( 8530364006697554384), -INT64_C( 6785738368543735910), -INT64_C( 4259669659578725301),  INT64_C( 2400221725814552185) },
       INT32_C(           0),
      { -INT64_C( 8530364006697554384), -INT64_C( 6785738368543735910),  INT64_C(                   0),  INT64_C( 2400221725814552185),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 8741758583524824506),  INT64_C(                   0) } },
    { UINT8_C( 82),
      { -INT64_C( 2614344370557477926), -INT64_C( 8226003661502701577), -INT64_C(  531807799883408699), -INT64_C( 8459380023214436359),
        -INT64_C( 1307808646788023463),  INT64_C(  947883788581006169), -INT64_C( 1296944380804509079), -INT64_C( 2585681351823440670) },
      {  INT64_C( 3796170387568074955),  INT64_C( 6914421764425410207), -INT64_C( 2437739340719740601), -INT64_C(  140862617136569684) },
       INT32_C(           0),
      {  INT64_C(                   0),  INT64_C( 6914421764425410207),  INT64_C(                   0),  INT64_C(                   0),
        -INT64_C( 1307808646788023463),  INT64_C(                   0), -INT64_C( 1296944380804509079),  INT64_C(                   0) } },
    { UINT8_C(229),
      { -INT64_C( 8443061019452274497), -INT64_C( 8017032782398043617),  INT64_C(  689795766016326148), -INT64_C( 9024011050066393900),
        -INT64_C( 5326134833440596274),  INT64_C( 7010001652292425102), -INT64_C( 5052154886858129343), -INT64_C( 8202077739271291929) },
      {  INT64_C( 8000414804015127739), -INT64_C( 7313724465795903037),  INT64_C( 2039664664097591727), -INT64_C(  134278666811676973) },
       INT32_C(           0),
      {  INT64_C( 8000414804015127739),  INT64_C(                   0),  INT64_C( 2039664664097591727),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C( 7010001652292425102), -INT64_C( 5052154886858129343), -INT64_C( 8202077739271291929) } },
    { UINT8_C( 59),
      { -INT64_C( 8923552407247786416), -INT64_C( 7936009823287344565), -INT64_C(  834647872939421130), -INT64_C(  239547658674983345),
        -INT64_C( 8231316499720825374), -INT64_C( 8166092272285665537), -INT64_C( 6815498396756699832),  INT64_C( 2869847689952120874) },
      { -INT64_C( 6655427840572971052),  INT64_C( 8919215450491976929), -INT64_C( 9154872091685562111), -INT64_C( 7415128657307957588) },
       INT32_C(           0),
      { -INT64_C( 6655427840572971052),  INT64_C( 8919215450491976929),  INT64_C(                   0), -INT64_C( 7415128657307957588),
        -INT64_C( 8231316499720825374), -INT64_C( 8166092272285665537),  INT64_C(                   0),  INT64_C(                   0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi64(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_inserti64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), test_vec[i].imm8, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m256i b = simde_test_x86_random_i64x4();
    int imm8 = simde_test_codegen_rand() & 1;
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_inserti64x4, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i32(2, imm8, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_insertf32x8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const float a[16];
    const float b[8];
    const int imm8;
    const float r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   271.14), SIMDE_FLOAT32_C(    18.39), SIMDE_FLOAT32_C(   871.18), SIMDE_FLOAT32_C(   355.90),
        SIMDE_FLOAT32_C(    81.96), SIMDE_FLOAT32_C(   592.94), SIMDE_FLOAT32_C(  -897.62), SIMDE_FLOAT32_C(   426.38),
        SIMDE_FLOAT32_C(  -120.60), SIMDE_FLOAT32_C(   937.95), SIMDE_FLOAT32_C(  -237.51), SIMDE_FLOAT32_C(   297.24),
        SIMDE_FLOAT32_C(   534.31), SIMDE_FLOAT32_C(  -905.98), SIMDE_FLOAT32_C(  -347.43), SIMDE_FLOAT32_C(  -998.14) },
      { SIMDE_FLOAT32_C(  -710.19), SIMDE_FLOAT32_C(   440.86), SIMDE_FLOAT32_C(  -892.51), SIMDE_FLOAT32_C(  -459.34),
        SIMDE_FLOAT32_C(   515.39), SIMDE_FLOAT32_C(   869.24), SIMDE_FLOAT32_C(  -556.25), SIMDE_FLOAT32_C(   169.39) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(  -710.19), SIMDE_FLOAT32_C(   440.86), SIMDE_FLOAT32_C(  -892.51), SIMDE_FLOAT32_C(  -459.34),
        SIMDE_FLOAT32_C(   515.39), SIMDE_FLOAT32_C(   869.24), SIMDE_FLOAT32_C(  -556.25), SIMDE_FLOAT32_C(   169.39),
        SIMDE_FLOAT32_C(  -120.60), SIMDE_FLOAT32_C(   937.95), SIMDE_FLOAT32_C(  -237.51), SIMDE_FLOAT32_C(   297.24),
        SIMDE_FLOAT32_C(   534.31), SIMDE_FLOAT32_C(  -905.98), SIMDE_FLOAT32_C(  -347.43), SIMDE_FLOAT32_C(  -998.14) } },
    { { SIMDE_FLOAT32_C(   829.09), SIMDE_FLOAT32_C(  -352.06), SIMDE_FLOAT32_C(  -145.19), SIMDE_FLOAT32_C(   875.46),
        SIMDE_FLOAT32_C(   -70.44), SIMDE_FLOAT32_C(  -187.99), SIMDE_FLOAT32_C(   146.60), SIMDE_FLOAT32_C(   947.95),
        SIMDE_FLOAT32_C(  -316.81), SIMDE_FLOAT32_C(  -497.50), SIMDE_FLOAT32_C(    29.90), SIMDE_FLOAT32_C(  -723.88),
        SIMDE_FLOAT32_C(  -395.12), SIMDE_FLOAT32_C(  -543.72), SIMDE_FLOAT32_C(   155.53), SIMDE_FLOAT32_C(  -457.17) },
      { SIMDE_FLOAT32_C(   218.77), SIMDE_FLOAT32_C(  -547.23), SIMDE_FLOAT32_C(  -922.86), SIMDE_FLOAT32_C(   312.79),
        SIMDE_FLOAT32_C(   105.34), SIMDE_FLOAT32_C(  -921.00), SIMDE_FLOAT32_C(   602.61), SIMDE_FLOAT32_C(  -453.80) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(   829.09), SIMDE_FLOAT32_C(  -352.06), SIMDE_FLOAT32_C(  -145.19), SIMDE_FLOAT32_C(   875.46),
        SIMDE_FLOAT32_C(   -70.44), SIMDE_FLOAT32_C(  -187.99), SIMDE_FLOAT32_C(   146.60), SIMDE_FLOAT32_C(   947.95),
        SIMDE_FLOAT32_C(   218.77), SIMDE_FLOAT32_C(  -547.23), SIMDE_FLOAT32_C(  -922.86), SIMDE_FLOAT32_C(   312.79),
        SIMDE_FLOAT32_C(   105.34), SIMDE_FLOAT32_C(  -921.00), SIMDE_FLOAT32_C(   602.61), SIMDE_FLOAT32_C(  -453.80) } },
    { { SIMDE_FLOAT32_C(  -856.74), SIMDE_FLOAT32_C(  -938.42), SIMDE_FLOAT32_C(  -944.27), SIMDE_FLOAT32_C(  -412.99),
        SIMDE_FLOAT32_C(   230.98), SIMDE_FLOAT32_C(  -594.73), SIMDE_FLOAT32_C(  -583.90), SIMDE_FLOAT32_C(   878.92),
        SIMDE_FLOAT32_C(   260.08), SIMDE_FLOAT32_C(  -708.44), SIMDE_FLOAT32_C(  -191.52), SIMDE_FLOAT32_C(  -927.90),
        SIMDE_FLOAT32_C(   438.16), SIMDE_FLOAT32_C(  -243.58), SIMDE_FLOAT32_C(  -244.72), SIMDE_FLOAT32_C(   940.66) },
      { SIMDE_FLOAT32_C(   786.33), SIMDE_FLOAT32_C(    31.41), SIMDE_FLOAT32_C(  -454.46), SIMDE_FLOAT32_C(  -757.39),
        SIMDE_FLOAT32_C(  -813.07), SIMDE_FLOAT32_C(    88.38), SIMDE_FLOAT32_C(   461.38), SIMDE_FLOAT32_C(  -360.30) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -856.74), SIMDE_FLOAT32_C(  -938.42), SIMDE_FLOAT32_C(  -944.27), SIMDE_FLOAT32_C(  -412.99),
        SIMDE_FLOAT32_C(   230.98), SIMDE_FLOAT32_C(  -594.73), SIMDE_FLOAT32_C(  -583.90), SIMDE_FLOAT32_C(   878.92),
        SIMDE_FLOAT32_C(   786.33), SIMDE_FLOAT32_C(    31.41), SIMDE_FLOAT32_C(  -454.46), SIMDE_FLOAT32_C(  -757.39),
        SIMDE_FLOAT32_C(  -813.07), SIMDE_FLOAT32_C(    88.38), SIMDE_FLOAT32_C(   461.38), SIMDE_FLOAT32_C(  -360.30) } },
    { { SIMDE_FLOAT32_C(  -225.83), SIMDE_FLOAT32_C(   745.04), SIMDE_FLOAT32_C(   244.52), SIMDE_FLOAT32_C(  -623.22),
        SIMDE_FLOAT32_C(  -708.77), SIMDE_FLOAT32_C(   431.01), SIMDE_FLOAT32_C(  -479.96), SIMDE_FLOAT32_C(  -647.18),
        SIMDE_FLOAT32_C(   486.74), SIMDE_FLOAT32_C(   107.05), SIMDE_FLOAT32_C(   583.79), SIMDE_FLOAT32_C(   892.01),
        SIMDE_FLOAT32_C(   523.15), SIMDE_FLOAT32_C(   462.71), SIMDE_FLOAT32_C(   152.10), SIMDE_FLOAT32_C(   814.72) },
      { SIMDE_FLOAT32_C(  -728.81), SIMDE_FLOAT32_C(   224.19), SIMDE_FLOAT32_C(   252.88), SIMDE_FLOAT32_C(    27.62),
        SIMDE_FLOAT32_C(   979.48), SIMDE_FLOAT32_C(   193.54), SIMDE_FLOAT32_C(  -186.06), SIMDE_FLOAT32_C(    10.88) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -225.83), SIMDE_FLOAT32_C(   745.04), SIMDE_FLOAT32_C(   244.52), SIMDE_FLOAT32_C(  -623.22),
        SIMDE_FLOAT32_C(  -708.77), SIMDE_FLOAT32_C(   431.01), SIMDE_FLOAT32_C(  -479.96), SIMDE_FLOAT32_C(  -647.18),
        SIMDE_FLOAT32_C(  -728.81), SIMDE_FLOAT32_C(   224.19), SIMDE_FLOAT32_C(   252.88), SIMDE_FLOAT32_C(    27.62),
        SIMDE_FLOAT32_C(   979.48), SIMDE_FLOAT32_C(   193.54), SIMDE_FLOAT32_C(  -186.06), SIMDE_FLOAT32_C(    10.88) } },
    { { SIMDE_FLOAT32_C(    56.55), SIMDE_FLOAT32_C(   197.82), SIMDE_FLOAT32_C(  -172.54), SIMDE_FLOAT32_C(  -482.07),
        SIMDE_FLOAT32_C(   837.52), SIMDE_FLOAT32_C(   992.97), SIMDE_FLOAT32_C(   292.10), SIMDE_FLOAT32_C(   582.56),
        SIMDE_FLOAT32_C(   237.49), SIMDE_FLOAT32_C(   668.88), SIMDE_FLOAT32_C(   873.79), SIMDE_FLOAT32_C(  -331.50),
        SIMDE_FLOAT32_C(  -811.08), SIMDE_FLOAT32_C(  -773.40), SIMDE_FLOAT32_C(  -844.75), SIMDE_FLOAT32_C(   295.97) },
      { SIMDE_FLOAT32_C(   810.40), SIMDE_FLOAT32_C(  -952.74), SIMDE_FLOAT32_C(  -180.87), SIMDE_FLOAT32_C(   273.11),
        SIMDE_FLOAT32_C(   199.36), SIMDE_FLOAT32_C(  -366.16), SIMDE_FLOAT32_C(   544.30), SIMDE_FLOAT32_C(  -576.45) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(   810.40), SIMDE_FLOAT32_C(  -952.74), SIMDE_FLOAT32_C(  -180.87), SIMDE_FLOAT32_C(   273.11),
        SIMDE_FLOAT32_C(   199.36), SIMDE_FLOAT32_C(  -366.16), SIMDE_FLOAT32_C(   544.30), SIMDE_FLOAT32_C(  -576.45),
        SIMDE_FLOAT32_C(   237.49), SIMDE_FLOAT32_C(   668.88), SIMDE_FLOAT32_C(   873.79), SIMDE_FLOAT32_C(  -331.50),
        SIMDE_FLOAT32_C(  -811.08), SIMDE_FLOAT32_C(  -773.40), SIMDE_FLOAT32_C(  -844.75), SIMDE_FLOAT32_C(   295.97) } },
    { { SIMDE_FLOAT32_C(  -428.09), SIMDE_FLOAT32_C(  -596.98), SIMDE_FLOAT32_C(    80.26), SIMDE_FLOAT32_C(   385.86),
        SIMDE_FLOAT32_C(   413.91), SIMDE_FLOAT32_C(  -180.65), SIMDE_FLOAT32_C(  -557.59), SIMDE_FLOAT32_C(  -388.27),
        SIMDE_FLOAT32_C(   646.80), SIMDE_FLOAT32_C(   -39.67), SIMDE_FLOAT32_C(  -550.75), SIMDE_FLOAT32_C(   639.78),
        SIMDE_FLOAT32_C(  -747.57), SIMDE_FLOAT32_C(  -968.20), SIMDE_FLOAT32_C(  -122.73), SIMDE_FLOAT32_C(   921.31) },
      { SIMDE_FLOAT32_C(   905.59), SIMDE_FLOAT32_C(   545.77), SIMDE_FLOAT32_C(  -889.77), SIMDE_FLOAT32_C(  -867.81),
        SIMDE_FLOAT32_C(   701.02), SIMDE_FLOAT32_C(   406.20), SIMDE_FLOAT32_C(   942.59), SIMDE_FLOAT32_C(   748.28) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -428.09), SIMDE_FLOAT32_C(  -596.98), SIMDE_FLOAT32_C(    80.26), SIMDE_FLOAT32_C(   385.86),
        SIMDE_FLOAT32_C(   413.91), SIMDE_FLOAT32_C(  -180.65), SIMDE_FLOAT32_C(  -557.59), SIMDE_FLOAT32_C(  -388.27),
        SIMDE_FLOAT32_C(   905.59), SIMDE_FLOAT32_C(   545.77), SIMDE_FLOAT32_C(  -889.77), SIMDE_FLOAT32_C(  -867.81),
        SIMDE_FLOAT32_C(   701.02), SIMDE_FLOAT32_C(   406.20), SIMDE_FLOAT32_C(   942.59), SIMDE_FLOAT32_C(   748.28) } },
    { { SIMDE_FLOAT32_C(   215.70), SIMDE_FLOAT32_C(   -52.36), SIMDE_FLOAT32_C(  -140.83), SIMDE_FLOAT32_C(  -240.00),
        SIMDE_FLOAT32_C(   371.19), SIMDE_FLOAT32_C(  -254.11), SIMDE_FLOAT32_C(   331.91), SIMDE_FLOAT32_C(   774.21),
        SIMDE_FLOAT32_C(   826.16), SIMDE_FLOAT32_C(  -282.23), SIMDE_FLOAT32_C(   188.12), SIMDE_FLOAT32_C(  -354.50),
        SIMDE_FLOAT32_C(   160.18), SIMDE_FLOAT32_C(   799.85), SIMDE_FLOAT32_C(  -707.70), SIMDE_FLOAT32_C(  -879.49) },
      { SIMDE_FLOAT32_C(  -750.91), SIMDE_FLOAT32_C(   932.08), SIMDE_FLOAT32_C(  -627.06), SIMDE_FLOAT32_C(  -719.11),
        SIMDE_FLOAT32_C(  -190.65), SIMDE_FLOAT32_C(  -705.75), SIMDE_FLOAT32_C(  -813.52), SIMDE_FLOAT32_C(  -644.88) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(  -750.91), SIMDE_FLOAT32_C(   932.08), SIMDE_FLOAT32_C(  -627.06), SIMDE_FLOAT32_C(  -719.11),
        SIMDE_FLOAT32_C(  -190.65), SIMDE_FLOAT32_C(  -705.75), SIMDE_FLOAT32_C(  -813.52), SIMDE_FLOAT32_C(  -644.88),
        SIMDE_FLOAT32_C(   826.16), SIMDE_FLOAT32_C(  -282.23), SIMDE_FLOAT32_C(   188.12), SIMDE_FLOAT32_C(  -354.50),
        SIMDE_FLOAT32_C(   160.18), SIMDE_FLOAT32_C(   799.85), SIMDE_FLOAT32_C(  -707.70), SIMDE_FLOAT32_C(  -879.49) } },
    { { SIMDE_FLOAT32_C(  -681.33), SIMDE_FLOAT32_C(  -943.86), SIMDE_FLOAT32_C(   810.67), SIMDE_FLOAT32_C(  -738.74),
        SIMDE_FLOAT32_C(   804.43), SIMDE_FLOAT32_C(  -964.00), SIMDE_FLOAT32_C(   476.96), SIMDE_FLOAT32_C(  -247.93),
        SIMDE_FLOAT32_C(  -104.83), SIMDE_FLOAT32_C(  -763.04), SIMDE_FLOAT32_C(  -876.74), SIMDE_FLOAT32_C(   641.06),
        SIMDE_FLOAT32_C(   568.88), SIMDE_FLOAT32_C(   897.47), SIMDE_FLOAT32_C(   467.22), SIMDE_FLOAT32_C(  -713.35) },
      { SIMDE_FLOAT32_C(    85.59), SIMDE_FLOAT32_C(  -887.28), SIMDE_FLOAT32_C(   446.83), SIMDE_FLOAT32_C(  -114.57),
        SIMDE_FLOAT32_C(  -594.98), SIMDE_FLOAT32_C(   567.34), SIMDE_FLOAT32_C(   134.53), SIMDE_FLOAT32_C(  -662.90) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(    85.59), SIMDE_FLOAT32_C(  -887.28), SIMDE_FLOAT32_C(   446.83), SIMDE_FLOAT32_C(  -114.57),
        SIMDE_FLOAT32_C(  -594.98), SIMDE_FLOAT32_C(   567.34), SIMDE_FLOAT32_C(   134.53), SIMDE_FLOAT32_C(  -662.90),
        SIMDE_FLOAT32_C(  -104.83), SIMDE_FLOAT32_C(  -763.04), SIMDE_FLOAT32_C(  -876.74), SIMDE_FLOAT32_C(   641.06),
        SIMDE_FLOAT32_C(   568.88), SIMDE_FLOAT32_C(   897.47), SIMDE_FLOAT32_C(   467.22), SIMDE_FLOAT32_C(  -713.35) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m256 b = simde_mm256_loadu_ps(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_assert_m512_close(simde_mm512_insertf32x8(a, b, 0), simde_mm512_loadu_ps(test_vec[i].r), 1); break;
      case 1: simde_assert_m512_close(simde_mm512_insertf32x8(a, b, 1), simde_mm512_loadu_ps(test_vec[i].r), 1); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m256 b = simde_test_x86_random_f32x8(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 0x01;
    simde__m512 r;
    SIMDE_CONSTIFY_2_(simde_mm512_insertf32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), imm8, a,b);

    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_insertf32x8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const float src[16];
    const simde__mmask16 k;
    const float a[16];
    const float b[8];
    const int imm8;
    const float r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   300.07), SIMDE_FLOAT32_C(   599.54), SIMDE_FLOAT32_C(   320.34), SIMDE_FLOAT32_C(   698.39),
        SIMDE_FLOAT32_C(  -154.22), SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(  -362.33), SIMDE_FLOAT32_C(   372.62),
        SIMDE_FLOAT32_C(   310.62), SIMDE_FLOAT32_C(   581.56), SIMDE_FLOAT32_C(  -243.70), SIMDE_FLOAT32_C(  -403.69),
        SIMDE_FLOAT32_C(   557.59), SIMDE_FLOAT32_C(   483.93), SIMDE_FLOAT32_C(  -282.13), SIMDE_FLOAT32_C(  -505.82) },
      UINT16_C( 8020),
      { SIMDE_FLOAT32_C(  -127.75), SIMDE_FLOAT32_C(  -168.03), SIMDE_FLOAT32_C(  -732.28), SIMDE_FLOAT32_C(   695.43),
        SIMDE_FLOAT32_C(  -514.09), SIMDE_FLOAT32_C(  -190.91), SIMDE_FLOAT32_C(   412.65), SIMDE_FLOAT32_C(   -54.40),
        SIMDE_FLOAT32_C(  -121.77), SIMDE_FLOAT32_C(   733.43), SIMDE_FLOAT32_C(   245.21), SIMDE_FLOAT32_C(   596.14),
        SIMDE_FLOAT32_C(  -710.34), SIMDE_FLOAT32_C(  -454.72), SIMDE_FLOAT32_C(   195.68), SIMDE_FLOAT32_C(   610.00) },
      { SIMDE_FLOAT32_C(  -756.33), SIMDE_FLOAT32_C(  -958.54), SIMDE_FLOAT32_C(   155.10), SIMDE_FLOAT32_C(  -118.66),
        SIMDE_FLOAT32_C(   414.08), SIMDE_FLOAT32_C(  -534.29), SIMDE_FLOAT32_C(  -537.10), SIMDE_FLOAT32_C(  -829.62) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(   300.07), SIMDE_FLOAT32_C(   599.54), SIMDE_FLOAT32_C(  -732.28), SIMDE_FLOAT32_C(   698.39),
        SIMDE_FLOAT32_C(  -514.09), SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(   412.65), SIMDE_FLOAT32_C(   372.62),
        SIMDE_FLOAT32_C(  -756.33), SIMDE_FLOAT32_C(  -958.54), SIMDE_FLOAT32_C(   155.10), SIMDE_FLOAT32_C(  -118.66),
        SIMDE_FLOAT32_C(   414.08), SIMDE_FLOAT32_C(   483.93), SIMDE_FLOAT32_C(  -282.13), SIMDE_FLOAT32_C(  -505.82) } },
    { { SIMDE_FLOAT32_C(  -979.51), SIMDE_FLOAT32_C(   654.31), SIMDE_FLOAT32_C(   779.88), SIMDE_FLOAT32_C(  -485.33),
        SIMDE_FLOAT32_C(   358.06), SIMDE_FLOAT32_C(   -67.37), SIMDE_FLOAT32_C(   386.93), SIMDE_FLOAT32_C(  -809.96),
        SIMDE_FLOAT32_C(   200.35), SIMDE_FLOAT32_C(    82.36), SIMDE_FLOAT32_C(  -324.06), SIMDE_FLOAT32_C(  -990.56),
        SIMDE_FLOAT32_C(  -504.99), SIMDE_FLOAT32_C(   621.55), SIMDE_FLOAT32_C(  -112.32), SIMDE_FLOAT32_C(  -771.56) },
      UINT16_C(31134),
      { SIMDE_FLOAT32_C(  -481.90), SIMDE_FLOAT32_C(   412.03), SIMDE_FLOAT32_C(   679.49), SIMDE_FLOAT32_C(  -871.91),
        SIMDE_FLOAT32_C(   655.70), SIMDE_FLOAT32_C(   720.95), SIMDE_FLOAT32_C(   283.19), SIMDE_FLOAT32_C(  -462.96),
        SIMDE_FLOAT32_C(   135.03), SIMDE_FLOAT32_C(   748.90), SIMDE_FLOAT32_C(    -0.06), SIMDE_FLOAT32_C(   305.41),
        SIMDE_FLOAT32_C(  -189.08), SIMDE_FLOAT32_C(    20.43), SIMDE_FLOAT32_C(   -40.27), SIMDE_FLOAT32_C(  -409.19) },
      { SIMDE_FLOAT32_C(   535.10), SIMDE_FLOAT32_C(  -682.21), SIMDE_FLOAT32_C(   523.44), SIMDE_FLOAT32_C(   -77.97),
        SIMDE_FLOAT32_C(  -492.17), SIMDE_FLOAT32_C(  -276.21), SIMDE_FLOAT32_C(  -995.62), SIMDE_FLOAT32_C(   183.77) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -979.51), SIMDE_FLOAT32_C(   412.03), SIMDE_FLOAT32_C(   679.49), SIMDE_FLOAT32_C(  -871.91),
        SIMDE_FLOAT32_C(   655.70), SIMDE_FLOAT32_C(   -67.37), SIMDE_FLOAT32_C(   386.93), SIMDE_FLOAT32_C(  -462.96),
        SIMDE_FLOAT32_C(   535.10), SIMDE_FLOAT32_C(    82.36), SIMDE_FLOAT32_C(  -324.06), SIMDE_FLOAT32_C(   -77.97),
        SIMDE_FLOAT32_C(  -492.17), SIMDE_FLOAT32_C(  -276.21), SIMDE_FLOAT32_C(  -995.62), SIMDE_FLOAT32_C(  -771.56) } },
    { { SIMDE_FLOAT32_C(  -500.61), SIMDE_FLOAT32_C(  -194.68), SIMDE_FLOAT32_C(   620.91), SIMDE_FLOAT32_C(  -272.17),
        SIMDE_FLOAT32_C(   672.07), SIMDE_FLOAT32_C(  -895.27), SIMDE_FLOAT32_C(   245.92), SIMDE_FLOAT32_C(    84.11),
        SIMDE_FLOAT32_C(   784.22), SIMDE_FLOAT32_C(   374.02), SIMDE_FLOAT32_C(  -260.19), SIMDE_FLOAT32_C(   505.17),
        SIMDE_FLOAT32_C(  -342.79), SIMDE_FLOAT32_C(   276.85), SIMDE_FLOAT32_C(  -359.80), SIMDE_FLOAT32_C(  -593.89) },
      UINT16_C(41694),
      { SIMDE_FLOAT32_C(   217.03), SIMDE_FLOAT32_C(   297.23), SIMDE_FLOAT32_C(   -94.66), SIMDE_FLOAT32_C(   807.84),
        SIMDE_FLOAT32_C(  -167.67), SIMDE_FLOAT32_C(   223.13), SIMDE_FLOAT32_C(   331.28), SIMDE_FLOAT32_C(   754.36),
        SIMDE_FLOAT32_C(   730.96), SIMDE_FLOAT32_C(  -944.93), SIMDE_FLOAT32_C(   758.74), SIMDE_FLOAT32_C(   -85.27),
        SIMDE_FLOAT32_C(  -211.69), SIMDE_FLOAT32_C(  -741.87), SIMDE_FLOAT32_C(   720.05), SIMDE_FLOAT32_C(  -590.78) },
      { SIMDE_FLOAT32_C(   -14.04), SIMDE_FLOAT32_C(   392.12), SIMDE_FLOAT32_C(  -486.06), SIMDE_FLOAT32_C(  -768.12),
        SIMDE_FLOAT32_C(  -523.78), SIMDE_FLOAT32_C(  -701.84), SIMDE_FLOAT32_C(   605.90), SIMDE_FLOAT32_C(   216.03) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -500.61), SIMDE_FLOAT32_C(   297.23), SIMDE_FLOAT32_C(   -94.66), SIMDE_FLOAT32_C(   807.84),
        SIMDE_FLOAT32_C(  -167.67), SIMDE_FLOAT32_C(  -895.27), SIMDE_FLOAT32_C(   331.28), SIMDE_FLOAT32_C(   754.36),
        SIMDE_FLOAT32_C(   784.22), SIMDE_FLOAT32_C(   392.12), SIMDE_FLOAT32_C(  -260.19), SIMDE_FLOAT32_C(   505.17),
        SIMDE_FLOAT32_C(  -342.79), SIMDE_FLOAT32_C(  -701.84), SIMDE_FLOAT32_C(  -359.80), SIMDE_FLOAT32_C(   216.03) } },
    { { SIMDE_FLOAT32_C(  -736.89), SIMDE_FLOAT32_C(  -507.11), SIMDE_FLOAT32_C(  -556.47), SIMDE_FLOAT32_C(  -330.78),
        SIMDE_FLOAT32_C(  -230.32), SIMDE_FLOAT32_C(  -610.86), SIMDE_FLOAT32_C(   886.26), SIMDE_FLOAT32_C(  -933.09),
        SIMDE_FLOAT32_C(   294.48), SIMDE_FLOAT32_C(   694.09), SIMDE_FLOAT32_C(  -100.77), SIMDE_FLOAT32_C(  -482.39),
        SIMDE_FLOAT32_C(    25.37), SIMDE_FLOAT32_C(  -346.41), SIMDE_FLOAT32_C(  -751.43), SIMDE_FLOAT32_C(    80.44) },
      UINT16_C(21804),
      { SIMDE_FLOAT32_C(   868.75), SIMDE_FLOAT32_C(  -329.54), SIMDE_FLOAT32_C(  -116.65), SIMDE_FLOAT32_C(  -722.03),
        SIMDE_FLOAT32_C(   656.41), SIMDE_FLOAT32_C(  -724.53), SIMDE_FLOAT32_C(  -208.09), SIMDE_FLOAT32_C(   888.29),
        SIMDE_FLOAT32_C(  -248.30), SIMDE_FLOAT32_C(    90.07), SIMDE_FLOAT32_C(   494.19), SIMDE_FLOAT32_C(   967.73),
        SIMDE_FLOAT32_C(  -106.60), SIMDE_FLOAT32_C(   757.30), SIMDE_FLOAT32_C(  -539.38), SIMDE_FLOAT32_C(   336.93) },
      { SIMDE_FLOAT32_C(  -573.48), SIMDE_FLOAT32_C(   230.30), SIMDE_FLOAT32_C(   726.07), SIMDE_FLOAT32_C(  -687.22),
        SIMDE_FLOAT32_C(   297.20), SIMDE_FLOAT32_C(    20.55), SIMDE_FLOAT32_C(  -993.12), SIMDE_FLOAT32_C(  -803.56) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -736.89), SIMDE_FLOAT32_C(  -507.11), SIMDE_FLOAT32_C(  -116.65), SIMDE_FLOAT32_C(  -722.03),
        SIMDE_FLOAT32_C(  -230.32), SIMDE_FLOAT32_C(  -724.53), SIMDE_FLOAT32_C(   886.26), SIMDE_FLOAT32_C(  -933.09),
        SIMDE_FLOAT32_C(  -573.48), SIMDE_FLOAT32_C(   694.09), SIMDE_FLOAT32_C(   726.07), SIMDE_FLOAT32_C(  -482.39),
        SIMDE_FLOAT32_C(   297.20), SIMDE_FLOAT32_C(  -346.41), SIMDE_FLOAT32_C(  -993.12), SIMDE_FLOAT32_C(    80.44) } },
    { { SIMDE_FLOAT32_C(    32.25), SIMDE_FLOAT32_C(  -149.97), SIMDE_FLOAT32_C(   786.73), SIMDE_FLOAT32_C(  -887.31),
        SIMDE_FLOAT32_C(   262.36), SIMDE_FLOAT32_C(   -49.96), SIMDE_FLOAT32_C(   981.44), SIMDE_FLOAT32_C(   932.81),
        SIMDE_FLOAT32_C(   833.39), SIMDE_FLOAT32_C(  -740.60), SIMDE_FLOAT32_C(   589.23), SIMDE_FLOAT32_C(  -891.13),
        SIMDE_FLOAT32_C(    51.31), SIMDE_FLOAT32_C(   477.52), SIMDE_FLOAT32_C(  -139.44), SIMDE_FLOAT32_C(  -858.62) },
      UINT16_C(34568),
      { SIMDE_FLOAT32_C(    34.78), SIMDE_FLOAT32_C(  -270.98), SIMDE_FLOAT32_C(   288.91), SIMDE_FLOAT32_C(  -628.29),
        SIMDE_FLOAT32_C(   155.54), SIMDE_FLOAT32_C(  -480.79), SIMDE_FLOAT32_C(  -902.22), SIMDE_FLOAT32_C(   468.32),
        SIMDE_FLOAT32_C(   816.42), SIMDE_FLOAT32_C(   118.32), SIMDE_FLOAT32_C(   475.20), SIMDE_FLOAT32_C(  -987.15),
        SIMDE_FLOAT32_C(  -343.52), SIMDE_FLOAT32_C(  -492.56), SIMDE_FLOAT32_C(  -137.12), SIMDE_FLOAT32_C(  -556.78) },
      { SIMDE_FLOAT32_C(  -379.87), SIMDE_FLOAT32_C(  -874.76), SIMDE_FLOAT32_C(   393.26), SIMDE_FLOAT32_C(  -398.43),
        SIMDE_FLOAT32_C(  -941.95), SIMDE_FLOAT32_C(   226.65), SIMDE_FLOAT32_C(  -139.02), SIMDE_FLOAT32_C(   647.28) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(    32.25), SIMDE_FLOAT32_C(  -149.97), SIMDE_FLOAT32_C(   786.73), SIMDE_FLOAT32_C(  -398.43),
        SIMDE_FLOAT32_C(   262.36), SIMDE_FLOAT32_C(   -49.96), SIMDE_FLOAT32_C(   981.44), SIMDE_FLOAT32_C(   932.81),
        SIMDE_FLOAT32_C(   816.42), SIMDE_FLOAT32_C(   118.32), SIMDE_FLOAT32_C(   475.20), SIMDE_FLOAT32_C(  -891.13),
        SIMDE_FLOAT32_C(    51.31), SIMDE_FLOAT32_C(   477.52), SIMDE_FLOAT32_C(  -139.44), SIMDE_FLOAT32_C(  -556.78) } },
    { { SIMDE_FLOAT32_C(   912.29), SIMDE_FLOAT32_C(   124.80), SIMDE_FLOAT32_C(  -803.92), SIMDE_FLOAT32_C(  -946.32),
        SIMDE_FLOAT32_C(  -903.49), SIMDE_FLOAT32_C(    24.37), SIMDE_FLOAT32_C(    88.46), SIMDE_FLOAT32_C(  -174.48),
        SIMDE_FLOAT32_C(  -686.71), SIMDE_FLOAT32_C(   460.17), SIMDE_FLOAT32_C(   981.06), SIMDE_FLOAT32_C(  -167.50),
        SIMDE_FLOAT32_C(   557.95), SIMDE_FLOAT32_C(   449.38), SIMDE_FLOAT32_C(  -351.08), SIMDE_FLOAT32_C(  -323.73) },
      UINT16_C(35670),
      { SIMDE_FLOAT32_C(   332.76), SIMDE_FLOAT32_C(   432.03), SIMDE_FLOAT32_C(   524.65), SIMDE_FLOAT32_C(   775.98),
        SIMDE_FLOAT32_C(  -947.84), SIMDE_FLOAT32_C(   649.89), SIMDE_FLOAT32_C(   169.23), SIMDE_FLOAT32_C(  -346.27),
        SIMDE_FLOAT32_C(   707.94), SIMDE_FLOAT32_C(  -604.12), SIMDE_FLOAT32_C(   514.71), SIMDE_FLOAT32_C(   355.22),
        SIMDE_FLOAT32_C(   731.40), SIMDE_FLOAT32_C(   427.00), SIMDE_FLOAT32_C(  -519.99), SIMDE_FLOAT32_C(   927.48) },
      { SIMDE_FLOAT32_C(   480.68), SIMDE_FLOAT32_C(  -423.48), SIMDE_FLOAT32_C(   -48.15), SIMDE_FLOAT32_C(  -430.86),
        SIMDE_FLOAT32_C(   402.04), SIMDE_FLOAT32_C(   265.14), SIMDE_FLOAT32_C(  -970.69), SIMDE_FLOAT32_C(   383.11) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(   912.29), SIMDE_FLOAT32_C(  -423.48), SIMDE_FLOAT32_C(   -48.15), SIMDE_FLOAT32_C(  -946.32),
        SIMDE_FLOAT32_C(   402.04), SIMDE_FLOAT32_C(    24.37), SIMDE_FLOAT32_C(  -970.69), SIMDE_FLOAT32_C(  -174.48),
        SIMDE_FLOAT32_C(   707.94), SIMDE_FLOAT32_C(  -604.12), SIMDE_FLOAT32_C(   981.06), SIMDE_FLOAT32_C(   355.22),
        SIMDE_FLOAT32_C(   557.95), SIMDE_FLOAT32_C(   449.38), SIMDE_FLOAT32_C(  -351.08), SIMDE_FLOAT32_C(   927.48) } },
    { { SIMDE_FLOAT32_C(   587.26), SIMDE_FLOAT32_C(  -167.51), SIMDE_FLOAT32_C(  -253.44), SIMDE_FLOAT32_C(  -736.47),
        SIMDE_FLOAT32_C(   757.07), SIMDE_FLOAT32_C(   408.33), SIMDE_FLOAT32_C(   596.29), SIMDE_FLOAT32_C(   189.10),
        SIMDE_FLOAT32_C(   -67.02), SIMDE_FLOAT32_C(   372.27), SIMDE_FLOAT32_C(   241.26), SIMDE_FLOAT32_C(  -417.13),
        SIMDE_FLOAT32_C(  -458.50), SIMDE_FLOAT32_C(   894.99), SIMDE_FLOAT32_C(  -709.19), SIMDE_FLOAT32_C(   -62.62) },
      UINT16_C(15641),
      { SIMDE_FLOAT32_C(  -331.22), SIMDE_FLOAT32_C(  -163.29), SIMDE_FLOAT32_C(  -873.96), SIMDE_FLOAT32_C(  -403.74),
        SIMDE_FLOAT32_C(  -682.61), SIMDE_FLOAT32_C(  -297.44), SIMDE_FLOAT32_C(   548.11), SIMDE_FLOAT32_C(  -113.47),
        SIMDE_FLOAT32_C(  -895.39), SIMDE_FLOAT32_C(  -186.75), SIMDE_FLOAT32_C(   -84.16), SIMDE_FLOAT32_C(   487.72),
        SIMDE_FLOAT32_C(   -89.11), SIMDE_FLOAT32_C(  -496.90), SIMDE_FLOAT32_C(  -679.79), SIMDE_FLOAT32_C(   657.44) },
      { SIMDE_FLOAT32_C(  -233.36), SIMDE_FLOAT32_C(  -922.72), SIMDE_FLOAT32_C(    65.77), SIMDE_FLOAT32_C(  -637.07),
        SIMDE_FLOAT32_C(   266.38), SIMDE_FLOAT32_C(   998.75), SIMDE_FLOAT32_C(   735.20), SIMDE_FLOAT32_C(  -492.36) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -331.22), SIMDE_FLOAT32_C(  -167.51), SIMDE_FLOAT32_C(  -253.44), SIMDE_FLOAT32_C(  -403.74),
        SIMDE_FLOAT32_C(  -682.61), SIMDE_FLOAT32_C(   408.33), SIMDE_FLOAT32_C(   596.29), SIMDE_FLOAT32_C(   189.10),
        SIMDE_FLOAT32_C(  -233.36), SIMDE_FLOAT32_C(   372.27), SIMDE_FLOAT32_C(    65.77), SIMDE_FLOAT32_C(  -637.07),
        SIMDE_FLOAT32_C(   266.38), SIMDE_FLOAT32_C(   998.75), SIMDE_FLOAT32_C(  -709.19), SIMDE_FLOAT32_C(   -62.62) } },
    { { SIMDE_FLOAT32_C(  -723.30), SIMDE_FLOAT32_C(  -597.36), SIMDE_FLOAT32_C(  -127.56), SIMDE_FLOAT32_C(   214.08),
        SIMDE_FLOAT32_C(   812.34), SIMDE_FLOAT32_C(  -481.53), SIMDE_FLOAT32_C(   882.87), SIMDE_FLOAT32_C(  -350.95),
        SIMDE_FLOAT32_C(  -355.49), SIMDE_FLOAT32_C(  -520.87), SIMDE_FLOAT32_C(   -33.57), SIMDE_FLOAT32_C(   347.07),
        SIMDE_FLOAT32_C(  -972.76), SIMDE_FLOAT32_C(   852.96), SIMDE_FLOAT32_C(   451.68), SIMDE_FLOAT32_C(  -159.52) },
      UINT16_C(34915),
      { SIMDE_FLOAT32_C(   751.37), SIMDE_FLOAT32_C(   271.91), SIMDE_FLOAT32_C(   259.61), SIMDE_FLOAT32_C(   408.81),
        SIMDE_FLOAT32_C(  -961.46), SIMDE_FLOAT32_C(   336.90), SIMDE_FLOAT32_C(  -525.42), SIMDE_FLOAT32_C(  -598.53),
        SIMDE_FLOAT32_C(  -396.72), SIMDE_FLOAT32_C(  -526.66), SIMDE_FLOAT32_C(  -863.33), SIMDE_FLOAT32_C(   110.92),
        SIMDE_FLOAT32_C(    54.96), SIMDE_FLOAT32_C(  -586.63), SIMDE_FLOAT32_C(   513.56), SIMDE_FLOAT32_C(   927.40) },
      { SIMDE_FLOAT32_C(   627.45), SIMDE_FLOAT32_C(   325.90), SIMDE_FLOAT32_C(  -554.13), SIMDE_FLOAT32_C(   510.32),
        SIMDE_FLOAT32_C(   974.95), SIMDE_FLOAT32_C(    90.38), SIMDE_FLOAT32_C(   989.45), SIMDE_FLOAT32_C(   -58.62) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(   751.37), SIMDE_FLOAT32_C(   271.91), SIMDE_FLOAT32_C(  -127.56), SIMDE_FLOAT32_C(   214.08),
        SIMDE_FLOAT32_C(   812.34), SIMDE_FLOAT32_C(   336.90), SIMDE_FLOAT32_C(  -525.42), SIMDE_FLOAT32_C(  -350.95),
        SIMDE_FLOAT32_C(  -355.49), SIMDE_FLOAT32_C(  -520.87), SIMDE_FLOAT32_C(   -33.57), SIMDE_FLOAT32_C(   510.32),
        SIMDE_FLOAT32_C(  -972.76), SIMDE_FLOAT32_C(   852.96), SIMDE_FLOAT32_C(   451.68), SIMDE_FLOAT32_C(   -58.62) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 src = simde_mm512_loadu_ps(test_vec[i].src);
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m256 b = simde_mm256_loadu_ps(test_vec[i].b);
    simde__m512 r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_insertf32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), test_vec[i].imm8 ,src, test_vec[i].k , a, b);
    simde_assert_m512_close(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512 src = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m256 b = simde_test_x86_random_f32x8(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 0x01;
    simde__m512 r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_insertf32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), imm8, src, k, a, b);
    simde_test_x86_write_f32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_insertf32x8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const float a[16];
    const float b[8];
    const int imm8;
    const float r[16];
  } test_vec[] = {
    { UINT16_C(57288),
      { SIMDE_FLOAT32_C(   806.34), SIMDE_FLOAT32_C(   897.43), SIMDE_FLOAT32_C(   825.27), SIMDE_FLOAT32_C(   331.34),
        SIMDE_FLOAT32_C(   452.51), SIMDE_FLOAT32_C(   233.49), SIMDE_FLOAT32_C(   372.02), SIMDE_FLOAT32_C(   428.47),
        SIMDE_FLOAT32_C(   372.19), SIMDE_FLOAT32_C(   758.73), SIMDE_FLOAT32_C(   138.04), SIMDE_FLOAT32_C(  -758.75),
        SIMDE_FLOAT32_C(  -449.41), SIMDE_FLOAT32_C(   860.85), SIMDE_FLOAT32_C(    30.97), SIMDE_FLOAT32_C(   268.02) },
      { SIMDE_FLOAT32_C(  -699.80), SIMDE_FLOAT32_C(   630.69), SIMDE_FLOAT32_C(  -975.74), SIMDE_FLOAT32_C(  -883.44),
        SIMDE_FLOAT32_C(  -726.56), SIMDE_FLOAT32_C(  -485.41), SIMDE_FLOAT32_C(   700.53), SIMDE_FLOAT32_C(   658.83) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   331.34),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   372.02), SIMDE_FLOAT32_C(   428.47),
        SIMDE_FLOAT32_C(  -699.80), SIMDE_FLOAT32_C(   630.69), SIMDE_FLOAT32_C(  -975.74), SIMDE_FLOAT32_C(  -883.44),
        SIMDE_FLOAT32_C(  -726.56), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   700.53), SIMDE_FLOAT32_C(   658.83) } },
    { UINT16_C(43443),
      { SIMDE_FLOAT32_C(  -243.08), SIMDE_FLOAT32_C(  -892.20), SIMDE_FLOAT32_C(  -900.56), SIMDE_FLOAT32_C(   508.10),
        SIMDE_FLOAT32_C(   914.14), SIMDE_FLOAT32_C(   996.87), SIMDE_FLOAT32_C(   333.37), SIMDE_FLOAT32_C(   245.48),
        SIMDE_FLOAT32_C(   449.38), SIMDE_FLOAT32_C(  -433.14), SIMDE_FLOAT32_C(  -382.50), SIMDE_FLOAT32_C(  -122.15),
        SIMDE_FLOAT32_C(   939.05), SIMDE_FLOAT32_C(  -623.77), SIMDE_FLOAT32_C(  -984.11), SIMDE_FLOAT32_C(  -819.70) },
      { SIMDE_FLOAT32_C(   -73.18), SIMDE_FLOAT32_C(   876.74), SIMDE_FLOAT32_C(   211.27), SIMDE_FLOAT32_C(  -805.15),
        SIMDE_FLOAT32_C(  -823.06), SIMDE_FLOAT32_C(  -158.04), SIMDE_FLOAT32_C(  -780.89), SIMDE_FLOAT32_C(  -706.50) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(   -73.18), SIMDE_FLOAT32_C(   876.74), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -823.06), SIMDE_FLOAT32_C(  -158.04), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -706.50),
        SIMDE_FLOAT32_C(   449.38), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -122.15),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -623.77), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -819.70) } },
    { UINT16_C(57896),
      { SIMDE_FLOAT32_C(  -225.77), SIMDE_FLOAT32_C(   167.24), SIMDE_FLOAT32_C(   880.92), SIMDE_FLOAT32_C(   491.13),
        SIMDE_FLOAT32_C(   924.16), SIMDE_FLOAT32_C(   988.73), SIMDE_FLOAT32_C(   590.57), SIMDE_FLOAT32_C(   432.27),
        SIMDE_FLOAT32_C(   902.87), SIMDE_FLOAT32_C(   587.43), SIMDE_FLOAT32_C(  -234.36), SIMDE_FLOAT32_C(   148.35),
        SIMDE_FLOAT32_C(    36.81), SIMDE_FLOAT32_C(   332.50), SIMDE_FLOAT32_C(   765.85), SIMDE_FLOAT32_C(   914.66) },
      { SIMDE_FLOAT32_C(   271.55), SIMDE_FLOAT32_C(  -857.92), SIMDE_FLOAT32_C(   930.55), SIMDE_FLOAT32_C(   451.85),
        SIMDE_FLOAT32_C(    68.91), SIMDE_FLOAT32_C(   807.28), SIMDE_FLOAT32_C(  -336.88), SIMDE_FLOAT32_C(   263.75) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   491.13),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   988.73), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -857.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   807.28), SIMDE_FLOAT32_C(  -336.88), SIMDE_FLOAT32_C(   263.75) } },
    { UINT16_C( 4413),
      { SIMDE_FLOAT32_C(  -722.28), SIMDE_FLOAT32_C(  -379.52), SIMDE_FLOAT32_C(  -783.44), SIMDE_FLOAT32_C(  -728.26),
        SIMDE_FLOAT32_C(   394.71), SIMDE_FLOAT32_C(   383.81), SIMDE_FLOAT32_C(  -847.34), SIMDE_FLOAT32_C(  -114.16),
        SIMDE_FLOAT32_C(   307.97), SIMDE_FLOAT32_C(  -858.61), SIMDE_FLOAT32_C(  -523.60), SIMDE_FLOAT32_C(  -259.77),
        SIMDE_FLOAT32_C(  -955.75), SIMDE_FLOAT32_C(  -936.17), SIMDE_FLOAT32_C(   505.87), SIMDE_FLOAT32_C(   192.60) },
      { SIMDE_FLOAT32_C(   100.64), SIMDE_FLOAT32_C(  -161.63), SIMDE_FLOAT32_C(   -41.55), SIMDE_FLOAT32_C(    15.30),
        SIMDE_FLOAT32_C(  -890.09), SIMDE_FLOAT32_C(   100.53), SIMDE_FLOAT32_C(   -54.15), SIMDE_FLOAT32_C(   561.76) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(  -722.28), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -783.44), SIMDE_FLOAT32_C(  -728.26),
        SIMDE_FLOAT32_C(   394.71), SIMDE_FLOAT32_C(   383.81), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   100.64), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -890.09), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(58307),
      { SIMDE_FLOAT32_C(   433.19), SIMDE_FLOAT32_C(  -262.65), SIMDE_FLOAT32_C(   729.96), SIMDE_FLOAT32_C(   -83.94),
        SIMDE_FLOAT32_C(    15.06), SIMDE_FLOAT32_C(  -649.56), SIMDE_FLOAT32_C(   132.62), SIMDE_FLOAT32_C(   286.80),
        SIMDE_FLOAT32_C(   745.15), SIMDE_FLOAT32_C(  -483.57), SIMDE_FLOAT32_C(   439.46), SIMDE_FLOAT32_C(  -369.01),
        SIMDE_FLOAT32_C(   824.39), SIMDE_FLOAT32_C(   580.85), SIMDE_FLOAT32_C(   107.39), SIMDE_FLOAT32_C(  -435.37) },
      { SIMDE_FLOAT32_C(   625.10), SIMDE_FLOAT32_C(   171.22), SIMDE_FLOAT32_C(  -929.50), SIMDE_FLOAT32_C(  -182.30),
        SIMDE_FLOAT32_C(  -728.14), SIMDE_FLOAT32_C(   -91.13), SIMDE_FLOAT32_C(   776.15), SIMDE_FLOAT32_C(   287.17) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(   625.10), SIMDE_FLOAT32_C(   171.22), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   776.15), SIMDE_FLOAT32_C(   287.17),
        SIMDE_FLOAT32_C(   745.15), SIMDE_FLOAT32_C(  -483.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   580.85), SIMDE_FLOAT32_C(   107.39), SIMDE_FLOAT32_C(  -435.37) } },
    { UINT16_C(62034),
      { SIMDE_FLOAT32_C(  -419.46), SIMDE_FLOAT32_C(    46.13), SIMDE_FLOAT32_C(   -13.86), SIMDE_FLOAT32_C(  -194.58),
        SIMDE_FLOAT32_C(  -520.68), SIMDE_FLOAT32_C(   723.49), SIMDE_FLOAT32_C(  -464.61), SIMDE_FLOAT32_C(   395.38),
        SIMDE_FLOAT32_C(  -261.45), SIMDE_FLOAT32_C(  -114.17), SIMDE_FLOAT32_C(  -472.00), SIMDE_FLOAT32_C(  -974.64),
        SIMDE_FLOAT32_C(  -369.02), SIMDE_FLOAT32_C(    44.43), SIMDE_FLOAT32_C(   464.82), SIMDE_FLOAT32_C(   261.97) },
      { SIMDE_FLOAT32_C(  -131.18), SIMDE_FLOAT32_C(    45.66), SIMDE_FLOAT32_C(  -630.65), SIMDE_FLOAT32_C(   433.45),
        SIMDE_FLOAT32_C(  -329.24), SIMDE_FLOAT32_C(   540.57), SIMDE_FLOAT32_C(   503.95), SIMDE_FLOAT32_C(   488.47) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    46.13), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -520.68), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -464.61), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    45.66), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -329.24), SIMDE_FLOAT32_C(   540.57), SIMDE_FLOAT32_C(   503.95), SIMDE_FLOAT32_C(   488.47) } },
    { UINT16_C( 8391),
      { SIMDE_FLOAT32_C(    99.60), SIMDE_FLOAT32_C(   431.59), SIMDE_FLOAT32_C(  -858.69), SIMDE_FLOAT32_C(   332.62),
        SIMDE_FLOAT32_C(  -987.87), SIMDE_FLOAT32_C(   187.43), SIMDE_FLOAT32_C(  -681.24), SIMDE_FLOAT32_C(  -182.44),
        SIMDE_FLOAT32_C(   666.76), SIMDE_FLOAT32_C(  -957.75), SIMDE_FLOAT32_C(   352.94), SIMDE_FLOAT32_C(    62.14),
        SIMDE_FLOAT32_C(  -219.20), SIMDE_FLOAT32_C(  -761.22), SIMDE_FLOAT32_C(   590.14), SIMDE_FLOAT32_C(  -193.84) },
      { SIMDE_FLOAT32_C(  -130.24), SIMDE_FLOAT32_C(  -365.43), SIMDE_FLOAT32_C(  -729.02), SIMDE_FLOAT32_C(  -868.28),
        SIMDE_FLOAT32_C(   503.39), SIMDE_FLOAT32_C(   316.64), SIMDE_FLOAT32_C(  -498.92), SIMDE_FLOAT32_C(   -63.16) },
       INT8_C(   1),
      { SIMDE_FLOAT32_C(    99.60), SIMDE_FLOAT32_C(   431.59), SIMDE_FLOAT32_C(  -858.69), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -681.24), SIMDE_FLOAT32_C(  -182.44),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   316.64), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(11730),
      { SIMDE_FLOAT32_C(   475.87), SIMDE_FLOAT32_C(   854.09), SIMDE_FLOAT32_C(  -146.40), SIMDE_FLOAT32_C(  -259.51),
        SIMDE_FLOAT32_C(   -46.31), SIMDE_FLOAT32_C(  -714.81), SIMDE_FLOAT32_C(  -118.21), SIMDE_FLOAT32_C(  -713.70),
        SIMDE_FLOAT32_C(  -702.68), SIMDE_FLOAT32_C(  -930.77), SIMDE_FLOAT32_C(  -394.94), SIMDE_FLOAT32_C(   114.88),
        SIMDE_FLOAT32_C(   735.98), SIMDE_FLOAT32_C(  -352.69), SIMDE_FLOAT32_C(  -532.18), SIMDE_FLOAT32_C(  -201.88) },
      { SIMDE_FLOAT32_C(   428.12), SIMDE_FLOAT32_C(  -293.40), SIMDE_FLOAT32_C(  -611.74), SIMDE_FLOAT32_C(  -765.72),
        SIMDE_FLOAT32_C(   576.35), SIMDE_FLOAT32_C(    22.82), SIMDE_FLOAT32_C(  -494.75), SIMDE_FLOAT32_C(   708.08) },
       INT8_C(   0),
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -293.40), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   576.35), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -494.75), SIMDE_FLOAT32_C(   708.08),
        SIMDE_FLOAT32_C(  -702.68), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -394.94), SIMDE_FLOAT32_C(   114.88),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -352.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m256 b = simde_mm256_loadu_ps(test_vec[i].b);
    simde__m512 r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_insertf32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), test_vec[i].imm8 , test_vec[i].k , a, b);
    simde_assert_m512_close(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m256 b = simde_test_x86_random_f32x8(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 0x01;
    simde__m512 r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_insertf32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_ps ()), imm8, k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_insertf64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[2];
    const int imm8;
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   323.23), SIMDE_FLOAT64_C(  -198.84), SIMDE_FLOAT64_C(   274.51), SIMDE_FLOAT64_C(   893.18),
        SIMDE_FLOAT64_C(  -795.95), SIMDE_FLOAT64_C(   650.94), SIMDE_FLOAT64_C(  -308.83), SIMDE_FLOAT64_C(   190.12) },
      { SIMDE_FLOAT64_C(  -807.59), SIMDE_FLOAT64_C(   247.88) },
       INT8_C(   0),
      { SIMDE_FLOAT64_C(  -807.59), SIMDE_FLOAT64_C(   247.88), SIMDE_FLOAT64_C(   274.51), SIMDE_FLOAT64_C(   893.18),
        SIMDE_FLOAT64_C(  -795.95), SIMDE_FLOAT64_C(   650.94), SIMDE_FLOAT64_C(  -308.83), SIMDE_FLOAT64_C(   190.12) } },
    { { SIMDE_FLOAT64_C(   492.09), SIMDE_FLOAT64_C(    62.93), SIMDE_FLOAT64_C(   590.35), SIMDE_FLOAT64_C(   310.79),
        SIMDE_FLOAT64_C(  -698.12), SIMDE_FLOAT64_C(   853.26), SIMDE_FLOAT64_C(  -196.77), SIMDE_FLOAT64_C(  -784.30) },
      { SIMDE_FLOAT64_C(   -76.52), SIMDE_FLOAT64_C(   412.09) },
       INT8_C(   3),
      { SIMDE_FLOAT64_C(   492.09), SIMDE_FLOAT64_C(    62.93), SIMDE_FLOAT64_C(   590.35), SIMDE_FLOAT64_C(   310.79),
        SIMDE_FLOAT64_C(  -698.12), SIMDE_FLOAT64_C(   853.26), SIMDE_FLOAT64_C(   -76.52), SIMDE_FLOAT64_C(   412.09) } },
    { { SIMDE_FLOAT64_C(   561.99), SIMDE_FLOAT64_C(  -579.79), SIMDE_FLOAT64_C(  -237.94), SIMDE_FLOAT64_C(   528.40),
        SIMDE_FLOAT64_C(   235.55), SIMDE_FLOAT64_C(   153.91), SIMDE_FLOAT64_C(   711.85), SIMDE_FLOAT64_C(  -889.98) },
      { SIMDE_FLOAT64_C(  -559.41), SIMDE_FLOAT64_C(    35.08) },
       INT8_C(   3),
      { SIMDE_FLOAT64_C(   561.99), SIMDE_FLOAT64_C(  -579.79), SIMDE_FLOAT64_C(  -237.94), SIMDE_FLOAT64_C(   528.40),
        SIMDE_FLOAT64_C(   235.55), SIMDE_FLOAT64_C(   153.91), SIMDE_FLOAT64_C(  -559.41), SIMDE_FLOAT64_C(    35.08) } },
    { { SIMDE_FLOAT64_C(   715.10), SIMDE_FLOAT64_C(   -71.73), SIMDE_FLOAT64_C(   115.22), SIMDE_FLOAT64_C(   366.04),
        SIMDE_FLOAT64_C(   619.44), SIMDE_FLOAT64_C(  -694.66), SIMDE_FLOAT64_C(   558.45), SIMDE_FLOAT64_C(  -132.68) },
      { SIMDE_FLOAT64_C(  -465.66), SIMDE_FLOAT64_C(    50.54) },
       INT8_C(   2),
      { SIMDE_FLOAT64_C(   715.10), SIMDE_FLOAT64_C(   -71.73), SIMDE_FLOAT64_C(   115.22), SIMDE_FLOAT64_C(   366.04),
        SIMDE_FLOAT64_C(  -465.66), SIMDE_FLOAT64_C(    50.54), SIMDE_FLOAT64_C(   558.45), SIMDE_FLOAT64_C(  -132.68) } },
    { { SIMDE_FLOAT64_C(  -875.31), SIMDE_FLOAT64_C(  -638.67), SIMDE_FLOAT64_C(  -767.87), SIMDE_FLOAT64_C(   977.96),
        SIMDE_FLOAT64_C(   164.56), SIMDE_FLOAT64_C(  -552.17), SIMDE_FLOAT64_C(   -98.56), SIMDE_FLOAT64_C(  -423.35) },
      { SIMDE_FLOAT64_C(   188.82), SIMDE_FLOAT64_C(  -536.57) },
       INT8_C(   3),
      { SIMDE_FLOAT64_C(  -875.31), SIMDE_FLOAT64_C(  -638.67), SIMDE_FLOAT64_C(  -767.87), SIMDE_FLOAT64_C(   977.96),
        SIMDE_FLOAT64_C(   164.56), SIMDE_FLOAT64_C(  -552.17), SIMDE_FLOAT64_C(   188.82), SIMDE_FLOAT64_C(  -536.57) } },
    { { SIMDE_FLOAT64_C(   950.87), SIMDE_FLOAT64_C(   991.83), SIMDE_FLOAT64_C(  -767.59), SIMDE_FLOAT64_C(   104.79),
        SIMDE_FLOAT64_C(   703.68), SIMDE_FLOAT64_C(  -657.57), SIMDE_FLOAT64_C(   545.38), SIMDE_FLOAT64_C(  -261.23) },
      { SIMDE_FLOAT64_C(   253.60), SIMDE_FLOAT64_C(   260.48) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(   950.87), SIMDE_FLOAT64_C(   991.83), SIMDE_FLOAT64_C(   253.60), SIMDE_FLOAT64_C(   260.48),
        SIMDE_FLOAT64_C(   703.68), SIMDE_FLOAT64_C(  -657.57), SIMDE_FLOAT64_C(   545.38), SIMDE_FLOAT64_C(  -261.23) } },
    { { SIMDE_FLOAT64_C(  -631.18), SIMDE_FLOAT64_C(  -373.48), SIMDE_FLOAT64_C(   286.47), SIMDE_FLOAT64_C(  -325.84),
        SIMDE_FLOAT64_C(  -815.03), SIMDE_FLOAT64_C(  -846.21), SIMDE_FLOAT64_C(   208.50), SIMDE_FLOAT64_C(   235.52) },
      { SIMDE_FLOAT64_C(  -915.96), SIMDE_FLOAT64_C(   333.19) },
       INT8_C(   0),
      { SIMDE_FLOAT64_C(  -915.96), SIMDE_FLOAT64_C(   333.19), SIMDE_FLOAT64_C(   286.47), SIMDE_FLOAT64_C(  -325.84),
        SIMDE_FLOAT64_C(  -815.03), SIMDE_FLOAT64_C(  -846.21), SIMDE_FLOAT64_C(   208.50), SIMDE_FLOAT64_C(   235.52) } },
    { { SIMDE_FLOAT64_C(  -683.83), SIMDE_FLOAT64_C(   311.15), SIMDE_FLOAT64_C(  -238.60), SIMDE_FLOAT64_C(  -235.99),
        SIMDE_FLOAT64_C(  -787.41), SIMDE_FLOAT64_C(   338.05), SIMDE_FLOAT64_C(   952.82), SIMDE_FLOAT64_C(  -323.98) },
      { SIMDE_FLOAT64_C(  -665.09), SIMDE_FLOAT64_C(   903.70) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(  -683.83), SIMDE_FLOAT64_C(   311.15), SIMDE_FLOAT64_C(  -665.09), SIMDE_FLOAT64_C(   903.70),
        SIMDE_FLOAT64_C(  -787.41), SIMDE_FLOAT64_C(   338.05), SIMDE_FLOAT64_C(   952.82), SIMDE_FLOAT64_C(  -323.98) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m128d b = simde_mm_loadu_pd(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_test_x86_assert_equal_f64x8(simde_mm512_insertf64x2(a, b, 0), simde_mm512_loadu_pd(test_vec[i].r), 1); break;
      case 1: simde_test_x86_assert_equal_f64x8(simde_mm512_insertf64x2(a, b, 1), simde_mm512_loadu_pd(test_vec[i].r), 1); break;
      case 2: simde_test_x86_assert_equal_f64x8(simde_mm512_insertf64x2(a, b, 2), simde_mm512_loadu_pd(test_vec[i].r), 1); break;
      case 3: simde_test_x86_assert_equal_f64x8(simde_mm512_insertf64x2(a, b, 3), simde_mm512_loadu_pd(test_vec[i].r), 1); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m128d b = simde_test_x86_random_f64x2(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 0x03;
    simde__m512d r;
    SIMDE_CONSTIFY_4_(simde_mm512_insertf64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), imm8, a, b);

    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_insertf64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float64 src[8];
    const simde__mmask8 k;
    const simde_float64 a[8];
    const simde_float64 b[2];
    const int imm8;
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   256.34), SIMDE_FLOAT64_C(   747.94), SIMDE_FLOAT64_C(   266.88), SIMDE_FLOAT64_C(   -66.89),
        SIMDE_FLOAT64_C(  -416.66), SIMDE_FLOAT64_C(  -772.24), SIMDE_FLOAT64_C(   273.91), SIMDE_FLOAT64_C(  -962.42) },
      UINT8_C(156),
      { SIMDE_FLOAT64_C(  -532.76), SIMDE_FLOAT64_C(  -816.17), SIMDE_FLOAT64_C(  -934.67), SIMDE_FLOAT64_C(  -311.09),
        SIMDE_FLOAT64_C(   818.42), SIMDE_FLOAT64_C(  -932.58), SIMDE_FLOAT64_C(  -257.35), SIMDE_FLOAT64_C(  -839.88) },
      { SIMDE_FLOAT64_C(    87.61), SIMDE_FLOAT64_C(  -376.95) },
       INT8_C(   0),
      { SIMDE_FLOAT64_C(   256.34), SIMDE_FLOAT64_C(   747.94), SIMDE_FLOAT64_C(  -934.67), SIMDE_FLOAT64_C(  -311.09),
        SIMDE_FLOAT64_C(   818.42), SIMDE_FLOAT64_C(  -772.24), SIMDE_FLOAT64_C(   273.91), SIMDE_FLOAT64_C(  -839.88) } },
    { { SIMDE_FLOAT64_C(   158.81), SIMDE_FLOAT64_C(  -752.21), SIMDE_FLOAT64_C(  -609.68), SIMDE_FLOAT64_C(    85.45),
        SIMDE_FLOAT64_C(  -826.84), SIMDE_FLOAT64_C(   307.47), SIMDE_FLOAT64_C(   692.59), SIMDE_FLOAT64_C(   633.54) },
      UINT8_C(229),
      { SIMDE_FLOAT64_C(   663.79), SIMDE_FLOAT64_C(   716.89), SIMDE_FLOAT64_C(   451.12), SIMDE_FLOAT64_C(   411.73),
        SIMDE_FLOAT64_C(   -16.23), SIMDE_FLOAT64_C(  -615.77), SIMDE_FLOAT64_C(   995.08), SIMDE_FLOAT64_C(   211.54) },
      { SIMDE_FLOAT64_C(   658.15), SIMDE_FLOAT64_C(  -967.34) },
       INT8_C(   3),
      { SIMDE_FLOAT64_C(   663.79), SIMDE_FLOAT64_C(  -752.21), SIMDE_FLOAT64_C(   451.12), SIMDE_FLOAT64_C(    85.45),
        SIMDE_FLOAT64_C(  -826.84), SIMDE_FLOAT64_C(  -615.77), SIMDE_FLOAT64_C(   658.15), SIMDE_FLOAT64_C(  -967.34) } },
    { { SIMDE_FLOAT64_C(  -874.61), SIMDE_FLOAT64_C(  -783.51), SIMDE_FLOAT64_C(   974.87), SIMDE_FLOAT64_C(  -185.71),
        SIMDE_FLOAT64_C(  -965.09), SIMDE_FLOAT64_C(  -957.71), SIMDE_FLOAT64_C(   556.94), SIMDE_FLOAT64_C(  -804.97) },
      UINT8_C(171),
      { SIMDE_FLOAT64_C(  -820.00), SIMDE_FLOAT64_C(   -96.62), SIMDE_FLOAT64_C(  -711.28), SIMDE_FLOAT64_C(  -572.21),
        SIMDE_FLOAT64_C(   293.69), SIMDE_FLOAT64_C(   374.17), SIMDE_FLOAT64_C(  -399.05), SIMDE_FLOAT64_C(  -398.84) },
      { SIMDE_FLOAT64_C(    66.76), SIMDE_FLOAT64_C(  -765.51) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(  -820.00), SIMDE_FLOAT64_C(   -96.62), SIMDE_FLOAT64_C(   974.87), SIMDE_FLOAT64_C(  -765.51),
        SIMDE_FLOAT64_C(  -965.09), SIMDE_FLOAT64_C(   374.17), SIMDE_FLOAT64_C(   556.94), SIMDE_FLOAT64_C(  -398.84) } },
    { { SIMDE_FLOAT64_C(  -269.45), SIMDE_FLOAT64_C(   951.38), SIMDE_FLOAT64_C(  -752.94), SIMDE_FLOAT64_C(  -857.71),
        SIMDE_FLOAT64_C(   -64.85), SIMDE_FLOAT64_C(  -368.70), SIMDE_FLOAT64_C(  -862.64), SIMDE_FLOAT64_C(  -853.31) },
      UINT8_C(169),
      { SIMDE_FLOAT64_C(  -829.98), SIMDE_FLOAT64_C(  -943.77), SIMDE_FLOAT64_C(  -585.17), SIMDE_FLOAT64_C(  -613.49),
        SIMDE_FLOAT64_C(  -968.90), SIMDE_FLOAT64_C(   229.12), SIMDE_FLOAT64_C(  -578.58), SIMDE_FLOAT64_C(  -926.61) },
      { SIMDE_FLOAT64_C(  -213.94), SIMDE_FLOAT64_C(  -383.56) },
       INT8_C(   3),
      { SIMDE_FLOAT64_C(  -829.98), SIMDE_FLOAT64_C(   951.38), SIMDE_FLOAT64_C(  -752.94), SIMDE_FLOAT64_C(  -613.49),
        SIMDE_FLOAT64_C(   -64.85), SIMDE_FLOAT64_C(   229.12), SIMDE_FLOAT64_C(  -862.64), SIMDE_FLOAT64_C(  -383.56) } },
    { { SIMDE_FLOAT64_C(   -33.94), SIMDE_FLOAT64_C(   519.82), SIMDE_FLOAT64_C(   492.02), SIMDE_FLOAT64_C(   393.84),
        SIMDE_FLOAT64_C(  -186.49), SIMDE_FLOAT64_C(  -133.81), SIMDE_FLOAT64_C(   994.79), SIMDE_FLOAT64_C(   414.67) },
      UINT8_C( 38),
      { SIMDE_FLOAT64_C(  -770.72), SIMDE_FLOAT64_C(  -789.38), SIMDE_FLOAT64_C(  -336.49), SIMDE_FLOAT64_C(  -819.34),
        SIMDE_FLOAT64_C(  -542.32), SIMDE_FLOAT64_C(  -194.21), SIMDE_FLOAT64_C(   115.82), SIMDE_FLOAT64_C(    88.98) },
      { SIMDE_FLOAT64_C(   -56.84), SIMDE_FLOAT64_C(   262.51) },
       INT8_C(   2),
      { SIMDE_FLOAT64_C(   -33.94), SIMDE_FLOAT64_C(  -789.38), SIMDE_FLOAT64_C(  -336.49), SIMDE_FLOAT64_C(   393.84),
        SIMDE_FLOAT64_C(  -186.49), SIMDE_FLOAT64_C(   262.51), SIMDE_FLOAT64_C(   994.79), SIMDE_FLOAT64_C(   414.67) } },
    { { SIMDE_FLOAT64_C(   113.18), SIMDE_FLOAT64_C(   318.74), SIMDE_FLOAT64_C(   793.24), SIMDE_FLOAT64_C(   499.69),
        SIMDE_FLOAT64_C(   349.83), SIMDE_FLOAT64_C(    22.36), SIMDE_FLOAT64_C(   921.10), SIMDE_FLOAT64_C(   423.23) },
      UINT8_C(166),
      { SIMDE_FLOAT64_C(  -462.46), SIMDE_FLOAT64_C(  -373.47), SIMDE_FLOAT64_C(  -225.52), SIMDE_FLOAT64_C(  -942.64),
        SIMDE_FLOAT64_C(  -881.46), SIMDE_FLOAT64_C(  -831.67), SIMDE_FLOAT64_C(  -129.13), SIMDE_FLOAT64_C(   -15.27) },
      { SIMDE_FLOAT64_C(  -836.88), SIMDE_FLOAT64_C(  -714.46) },
       INT8_C(   2),
      { SIMDE_FLOAT64_C(   113.18), SIMDE_FLOAT64_C(  -373.47), SIMDE_FLOAT64_C(  -225.52), SIMDE_FLOAT64_C(   499.69),
        SIMDE_FLOAT64_C(   349.83), SIMDE_FLOAT64_C(  -714.46), SIMDE_FLOAT64_C(   921.10), SIMDE_FLOAT64_C(   -15.27) } },
    { { SIMDE_FLOAT64_C(  -607.60), SIMDE_FLOAT64_C(  -503.84), SIMDE_FLOAT64_C(   581.19), SIMDE_FLOAT64_C(  -426.93),
        SIMDE_FLOAT64_C(   -46.16), SIMDE_FLOAT64_C(  -613.02), SIMDE_FLOAT64_C(   688.89), SIMDE_FLOAT64_C(  -957.18) },
      UINT8_C( 69),
      { SIMDE_FLOAT64_C(   -48.61), SIMDE_FLOAT64_C(   421.23), SIMDE_FLOAT64_C(  -556.68), SIMDE_FLOAT64_C(  -729.87),
        SIMDE_FLOAT64_C(   214.48), SIMDE_FLOAT64_C(   943.00), SIMDE_FLOAT64_C(   619.96), SIMDE_FLOAT64_C(  -763.16) },
      { SIMDE_FLOAT64_C(   864.10), SIMDE_FLOAT64_C(    43.19) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(   -48.61), SIMDE_FLOAT64_C(  -503.84), SIMDE_FLOAT64_C(   864.10), SIMDE_FLOAT64_C(  -426.93),
        SIMDE_FLOAT64_C(   -46.16), SIMDE_FLOAT64_C(  -613.02), SIMDE_FLOAT64_C(   619.96), SIMDE_FLOAT64_C(  -957.18) } },
    { { SIMDE_FLOAT64_C(  -598.36), SIMDE_FLOAT64_C(   669.71), SIMDE_FLOAT64_C(  -180.25), SIMDE_FLOAT64_C(  -541.00),
        SIMDE_FLOAT64_C(   788.26), SIMDE_FLOAT64_C(   -11.92), SIMDE_FLOAT64_C(   329.87), SIMDE_FLOAT64_C(  -227.01) },
      UINT8_C(128),
      { SIMDE_FLOAT64_C(   615.42), SIMDE_FLOAT64_C(   690.67), SIMDE_FLOAT64_C(   543.60), SIMDE_FLOAT64_C(  -888.42),
        SIMDE_FLOAT64_C(   271.86), SIMDE_FLOAT64_C(  -883.33), SIMDE_FLOAT64_C(    65.42), SIMDE_FLOAT64_C(   658.84) },
      { SIMDE_FLOAT64_C(   805.56), SIMDE_FLOAT64_C(   108.23) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(  -598.36), SIMDE_FLOAT64_C(   669.71), SIMDE_FLOAT64_C(  -180.25), SIMDE_FLOAT64_C(  -541.00),
        SIMDE_FLOAT64_C(   788.26), SIMDE_FLOAT64_C(   -11.92), SIMDE_FLOAT64_C(   329.87), SIMDE_FLOAT64_C(   658.84) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d src = simde_mm512_loadu_pd(test_vec[i].src);
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m128d b = simde_mm_loadu_pd(test_vec[i].b);
    simde__m512d r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_insertf64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), test_vec[i].imm8, src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512d src = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m128d b = simde_test_x86_random_f64x2(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 0x03;
    simde__m512d r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_insertf64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), imm8, src, k, a, b);

    simde_test_x86_write_f64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_insertf64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const simde_float64 a[8];
    const simde_float64 b[2];
    const int imm8;
    const simde_float64 r[8];
  } test_vec[] = {
    { UINT8_C(203),
      { SIMDE_FLOAT64_C(  -102.41), SIMDE_FLOAT64_C(  -984.50), SIMDE_FLOAT64_C(   378.83), SIMDE_FLOAT64_C(   671.25),
        SIMDE_FLOAT64_C(    70.43), SIMDE_FLOAT64_C(   739.49), SIMDE_FLOAT64_C(   560.20), SIMDE_FLOAT64_C(   853.15) },
      { SIMDE_FLOAT64_C(  -890.75), SIMDE_FLOAT64_C(   471.35) },
       INT8_C(   0),
      { SIMDE_FLOAT64_C(  -890.75), SIMDE_FLOAT64_C(   471.35), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   671.25),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   560.20), SIMDE_FLOAT64_C(   853.15) } },
    { UINT8_C( 23),
      { SIMDE_FLOAT64_C(  -158.79), SIMDE_FLOAT64_C(  -502.09), SIMDE_FLOAT64_C(   -64.26), SIMDE_FLOAT64_C(   685.47),
        SIMDE_FLOAT64_C(   -11.74), SIMDE_FLOAT64_C(  -667.75), SIMDE_FLOAT64_C(  -737.68), SIMDE_FLOAT64_C(   408.47) },
      { SIMDE_FLOAT64_C(   266.07), SIMDE_FLOAT64_C(   988.14) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(  -158.79), SIMDE_FLOAT64_C(  -502.09), SIMDE_FLOAT64_C(   266.07), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(   -11.74), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C( 15),
      { SIMDE_FLOAT64_C(   885.97), SIMDE_FLOAT64_C(   703.63), SIMDE_FLOAT64_C(  -645.51), SIMDE_FLOAT64_C(   -85.07),
        SIMDE_FLOAT64_C(  -833.84), SIMDE_FLOAT64_C(    40.04), SIMDE_FLOAT64_C(    30.47), SIMDE_FLOAT64_C(    63.75) },
      { SIMDE_FLOAT64_C(    55.54), SIMDE_FLOAT64_C(  -590.69) },
       INT8_C(   2),
      { SIMDE_FLOAT64_C(   885.97), SIMDE_FLOAT64_C(   703.63), SIMDE_FLOAT64_C(  -645.51), SIMDE_FLOAT64_C(   -85.07),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C( 83),
      { SIMDE_FLOAT64_C(  -851.20), SIMDE_FLOAT64_C(  -704.81), SIMDE_FLOAT64_C(   979.12), SIMDE_FLOAT64_C(  -741.95),
        SIMDE_FLOAT64_C(   766.54), SIMDE_FLOAT64_C(  -423.03), SIMDE_FLOAT64_C(   639.69), SIMDE_FLOAT64_C(  -392.24) },
      { SIMDE_FLOAT64_C(    74.89), SIMDE_FLOAT64_C(  -424.57) },
       INT8_C(   3),
      { SIMDE_FLOAT64_C(  -851.20), SIMDE_FLOAT64_C(  -704.81), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(   766.54), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(    74.89), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(193),
      { SIMDE_FLOAT64_C(   -92.32), SIMDE_FLOAT64_C(  -444.45), SIMDE_FLOAT64_C(   471.62), SIMDE_FLOAT64_C(  -826.24),
        SIMDE_FLOAT64_C(  -456.31), SIMDE_FLOAT64_C(  -408.85), SIMDE_FLOAT64_C(  -263.04), SIMDE_FLOAT64_C(  -570.34) },
      { SIMDE_FLOAT64_C(  -705.22), SIMDE_FLOAT64_C(    91.45) },
       INT8_C(   2),
      { SIMDE_FLOAT64_C(   -92.32), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -263.04), SIMDE_FLOAT64_C(  -570.34) } },
    { UINT8_C(  5),
      { SIMDE_FLOAT64_C(  -868.52), SIMDE_FLOAT64_C(  -624.93), SIMDE_FLOAT64_C(   524.70), SIMDE_FLOAT64_C(   187.02),
        SIMDE_FLOAT64_C(  -215.63), SIMDE_FLOAT64_C(  -740.31), SIMDE_FLOAT64_C(   312.99), SIMDE_FLOAT64_C(   -66.83) },
      { SIMDE_FLOAT64_C(  -445.12), SIMDE_FLOAT64_C(   292.10) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(  -868.52), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -445.12), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(174),
      { SIMDE_FLOAT64_C(   869.07), SIMDE_FLOAT64_C(  -169.09), SIMDE_FLOAT64_C(   -70.82), SIMDE_FLOAT64_C(   -56.04),
        SIMDE_FLOAT64_C(   406.34), SIMDE_FLOAT64_C(   222.41), SIMDE_FLOAT64_C(     7.11), SIMDE_FLOAT64_C(  -685.98) },
      { SIMDE_FLOAT64_C(   777.96), SIMDE_FLOAT64_C(  -521.27) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -169.09), SIMDE_FLOAT64_C(   777.96), SIMDE_FLOAT64_C(  -521.27),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   222.41), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -685.98) } },
    { UINT8_C( 92),
      { SIMDE_FLOAT64_C(    69.88), SIMDE_FLOAT64_C(   224.74), SIMDE_FLOAT64_C(  -248.69), SIMDE_FLOAT64_C(   364.66),
        SIMDE_FLOAT64_C(  -683.82), SIMDE_FLOAT64_C(  -904.10), SIMDE_FLOAT64_C(   825.60), SIMDE_FLOAT64_C(  -552.33) },
      { SIMDE_FLOAT64_C(  -529.04), SIMDE_FLOAT64_C(   350.30) },
       INT8_C(   1),
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -529.04), SIMDE_FLOAT64_C(   350.30),
        SIMDE_FLOAT64_C(  -683.82), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   825.60), SIMDE_FLOAT64_C(     0.00) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m128d b = simde_mm_loadu_pd(test_vec[i].b);
    simde__m512d r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_insertf64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), test_vec[i].imm8, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    simde__m128d b = simde_test_x86_random_f64x2(SIMDE_FLOAT64_C(-1000.0), SIMDE_FLOAT64_C(1000.0));
    int imm8 = simde_test_codegen_rand() & 0x03;
    simde__m512d r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_insertf64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_pd ()), imm8, k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_inserti32x8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    const int32_t b[8];
    const int imm8;
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1918935898),  INT32_C(  1646012911), -INT32_C(  1672505785), -INT32_C(   322953844), -INT32_C(   193886883), -INT32_C(  1435658764),  INT32_C(   271866963), -INT32_C(   615049931),
         INT32_C(  1634268786),  INT32_C(  1724089631), -INT32_C(  1493036261), -INT32_C(  1902919120),  INT32_C(  1015153736), -INT32_C(   303566951), -INT32_C(  2130896053), -INT32_C(  1554295759) },
      {  INT32_C(  1761920074),  INT32_C(  1691338569),  INT32_C(   185389786), -INT32_C(   593911916),  INT32_C(  1025055652),  INT32_C(  1428815882),  INT32_C(  1289103387), -INT32_C(   940625540) },
       INT8_C(   1),
      { -INT32_C(  1918935898),  INT32_C(  1646012911), -INT32_C(  1672505785), -INT32_C(   322953844), -INT32_C(   193886883), -INT32_C(  1435658764),  INT32_C(   271866963), -INT32_C(   615049931),
         INT32_C(  1761920074),  INT32_C(  1691338569),  INT32_C(   185389786), -INT32_C(   593911916),  INT32_C(  1025055652),  INT32_C(  1428815882),  INT32_C(  1289103387), -INT32_C(   940625540) } },
    { { -INT32_C(  1153552141), -INT32_C(   761945344),  INT32_C(  1315348655),  INT32_C(  1425163065),  INT32_C(  1549676636), -INT32_C(  2106084262), -INT32_C(  1140865910), -INT32_C(  1481521485),
        -INT32_C(   161288202),  INT32_C(  1103689618), -INT32_C(   779079784), -INT32_C(   836337038),  INT32_C(   220890290), -INT32_C(  1030774472),  INT32_C(   427724389),  INT32_C(  1270886228) },
      { -INT32_C(  1321131490), -INT32_C(  1309537767), -INT32_C(  1417444807), -INT32_C(  1216763643),  INT32_C(  1724162861), -INT32_C(  1440197564),  INT32_C(   935569122), -INT32_C(   192773163) },
       INT8_C(   1),
      { -INT32_C(  1153552141), -INT32_C(   761945344),  INT32_C(  1315348655),  INT32_C(  1425163065),  INT32_C(  1549676636), -INT32_C(  2106084262), -INT32_C(  1140865910), -INT32_C(  1481521485),
        -INT32_C(  1321131490), -INT32_C(  1309537767), -INT32_C(  1417444807), -INT32_C(  1216763643),  INT32_C(  1724162861), -INT32_C(  1440197564),  INT32_C(   935569122), -INT32_C(   192773163) } },
    { { -INT32_C(   843143741),  INT32_C(   436629655), -INT32_C(  1675644429), -INT32_C(   842410454), -INT32_C(   284020837), -INT32_C(    19809192),  INT32_C(    47450239),  INT32_C(  1319618698),
         INT32_C(    68904301), -INT32_C(   937549099), -INT32_C(    26985005), -INT32_C(  1345638892), -INT32_C(  1247879843),  INT32_C(   414478233),  INT32_C(    35293304), -INT32_C(  1118781104) },
      { -INT32_C(    54432729),  INT32_C(  1640358030),  INT32_C(   828320029), -INT32_C(  1243600041), -INT32_C(  1570079224),  INT32_C(  1723473646), -INT32_C(   160901722), -INT32_C(  1112295018) },
       INT8_C(   1),
      { -INT32_C(   843143741),  INT32_C(   436629655), -INT32_C(  1675644429), -INT32_C(   842410454), -INT32_C(   284020837), -INT32_C(    19809192),  INT32_C(    47450239),  INT32_C(  1319618698),
        -INT32_C(    54432729),  INT32_C(  1640358030),  INT32_C(   828320029), -INT32_C(  1243600041), -INT32_C(  1570079224),  INT32_C(  1723473646), -INT32_C(   160901722), -INT32_C(  1112295018) } },
    { {  INT32_C(  1437842037), -INT32_C(  1468918657), -INT32_C(  1627347852),  INT32_C(    61322628),  INT32_C(  1056000287), -INT32_C(   639346940),  INT32_C(  2020596671),  INT32_C(    60632462),
         INT32_C(  1717063911), -INT32_C(   653341851), -INT32_C(   210235793), -INT32_C(   470409277),  INT32_C(  1830938473), -INT32_C(    28965570),  INT32_C(  1870050784), -INT32_C(   915270430) },
      { -INT32_C(   919614620),  INT32_C(    94518934),  INT32_C(   284695116), -INT32_C(  1544294598),  INT32_C(   353375446), -INT32_C(    99396071), -INT32_C(   295073525),  INT32_C(    45603741) },
       INT8_C(   0),
      { -INT32_C(   919614620),  INT32_C(    94518934),  INT32_C(   284695116), -INT32_C(  1544294598),  INT32_C(   353375446), -INT32_C(    99396071), -INT32_C(   295073525),  INT32_C(    45603741),
         INT32_C(  1717063911), -INT32_C(   653341851), -INT32_C(   210235793), -INT32_C(   470409277),  INT32_C(  1830938473), -INT32_C(    28965570),  INT32_C(  1870050784), -INT32_C(   915270430) } },
    { {  INT32_C(   624741351), -INT32_C(  2005843602),  INT32_C(   700612922), -INT32_C(  1996528268), -INT32_C(   878570379), -INT32_C(  1311269849), -INT32_C(   531708667),  INT32_C(  1669812348),
        -INT32_C(  1970748644),  INT32_C(  1058208261), -INT32_C(   261565061), -INT32_C(  1334286534), -INT32_C(  1552213380), -INT32_C(  1152101706), -INT32_C(  1801673961),  INT32_C(   267854835) },
      { -INT32_C(   342261530), -INT32_C(   181752710), -INT32_C(  1142582655),  INT32_C(  1986748154),  INT32_C(   790226808),  INT32_C(  1374318137),  INT32_C(    82150929), -INT32_C(  1877680983) },
       INT8_C(   0),
      { -INT32_C(   342261530), -INT32_C(   181752710), -INT32_C(  1142582655),  INT32_C(  1986748154),  INT32_C(   790226808),  INT32_C(  1374318137),  INT32_C(    82150929), -INT32_C(  1877680983),
        -INT32_C(  1970748644),  INT32_C(  1058208261), -INT32_C(   261565061), -INT32_C(  1334286534), -INT32_C(  1552213380), -INT32_C(  1152101706), -INT32_C(  1801673961),  INT32_C(   267854835) } },
    { {  INT32_C(  1507228589),  INT32_C(   953863334),  INT32_C(   254973617), -INT32_C(   393697279),  INT32_C(   807581634),  INT32_C(   675378081),  INT32_C(   886130008),  INT32_C(   126968153),
        -INT32_C(  2090833955), -INT32_C(   440714445), -INT32_C(   755700015),  INT32_C(  1488682134), -INT32_C(   712450765), -INT32_C(  1459762864),  INT32_C(  1759301135),  INT32_C(   225406256) },
      {  INT32_C(   143708373), -INT32_C(   588428533), -INT32_C(   793845191), -INT32_C(  1842845346), -INT32_C(  1771589306), -INT32_C(  1992399750),  INT32_C(  1660033842),  INT32_C(  1567580552) },
       INT8_C(   1),
      {  INT32_C(  1507228589),  INT32_C(   953863334),  INT32_C(   254973617), -INT32_C(   393697279),  INT32_C(   807581634),  INT32_C(   675378081),  INT32_C(   886130008),  INT32_C(   126968153),
         INT32_C(   143708373), -INT32_C(   588428533), -INT32_C(   793845191), -INT32_C(  1842845346), -INT32_C(  1771589306), -INT32_C(  1992399750),  INT32_C(  1660033842),  INT32_C(  1567580552) } },
    { {  INT32_C(  1262249727),  INT32_C(   897849427),  INT32_C(   815027399),  INT32_C(   779560573), -INT32_C(   240644723),  INT32_C(  1730359884), -INT32_C(  2047900124), -INT32_C(   172536331),
         INT32_C(   104919987), -INT32_C(   750992372), -INT32_C(  1778135015), -INT32_C(  2084275466), -INT32_C(   730567544), -INT32_C(  1036281954),  INT32_C(   323496477),  INT32_C(   705232759) },
      { -INT32_C(    30324494),  INT32_C(   634481932),  INT32_C(   867947837), -INT32_C(   675905713), -INT32_C(  1968493844), -INT32_C(   548542783), -INT32_C(  1997368047), -INT32_C(  2035025260) },
       INT8_C(   0),
      { -INT32_C(    30324494),  INT32_C(   634481932),  INT32_C(   867947837), -INT32_C(   675905713), -INT32_C(  1968493844), -INT32_C(   548542783), -INT32_C(  1997368047), -INT32_C(  2035025260),
         INT32_C(   104919987), -INT32_C(   750992372), -INT32_C(  1778135015), -INT32_C(  2084275466), -INT32_C(   730567544), -INT32_C(  1036281954),  INT32_C(   323496477),  INT32_C(   705232759) } },
    { {  INT32_C(  1364100580),  INT32_C(   730756182), -INT32_C(  1350844113), -INT32_C(  1583656329), -INT32_C(   463329794),  INT32_C(   133513586),  INT32_C(   765165107),  INT32_C(   359604785),
        -INT32_C(    43598169),  INT32_C(  1630139442),  INT32_C(   739288245), -INT32_C(   187847690),  INT32_C(  1138307025), -INT32_C(  1538535823),  INT32_C(  2110973516), -INT32_C(  1349369592) },
      {  INT32_C(   833485055), -INT32_C(  1584146708),  INT32_C(  1892524922),  INT32_C(   543529551),  INT32_C(   996359882),  INT32_C(  1507831308), -INT32_C(  1646874219), -INT32_C(   229807629) },
       INT8_C(   1),
      {  INT32_C(  1364100580),  INT32_C(   730756182), -INT32_C(  1350844113), -INT32_C(  1583656329), -INT32_C(   463329794),  INT32_C(   133513586),  INT32_C(   765165107),  INT32_C(   359604785),
         INT32_C(   833485055), -INT32_C(  1584146708),  INT32_C(  1892524922),  INT32_C(   543529551),  INT32_C(   996359882),  INT32_C(  1507831308), -INT32_C(  1646874219), -INT32_C(   229807629) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi32(test_vec[i].b);
    simde__m512i r;
    switch(test_vec[i].imm8) {
      case 0: r = simde_mm512_inserti32x8(a, b, 0); break;
      case 1: r = simde_mm512_inserti32x8(a, b, 1); break;
      default: HEDLEY_UNREACHABLE(); r = simde_mm512_setzero_epi32(); break;
    }
    simde_assert_m512i_i32(r, == ,simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m256i b = simde_test_x86_random_i32x8();
    int imm8 = simde_test_codegen_rand() & 0x01;
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_inserti32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_epi32 ()), imm8, a,b);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_inserti32x8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[8];
    const int imm8;
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1639741622), -INT32_C(  1034527332),  INT32_C(   252024761),  INT32_C(   219231863),  INT32_C(    34477431),  INT32_C(  1024205182),  INT32_C(  2139059556),  INT32_C(   449668048),
        -INT32_C(  2034757142),  INT32_C(   675810926),  INT32_C(   490163622), -INT32_C(    97826685), -INT32_C(   604161699), -INT32_C(  1105655462),  INT32_C(  1329436799), -INT32_C(   429323269) },
      UINT16_C( 8732),
      { -INT32_C(  1255110036), -INT32_C(   385689934),  INT32_C(   523405044),  INT32_C(  2102955904), -INT32_C(  2071481749), -INT32_C(  1373895055), -INT32_C(  1095165867),  INT32_C(  1793119742),
         INT32_C(   304025952), -INT32_C(   604232985),  INT32_C(   687484584),  INT32_C(   681988797),  INT32_C(  1990995205), -INT32_C(  2010855117), -INT32_C(   549003551),  INT32_C(   323561395) },
      {  INT32_C(   522611000),  INT32_C(   872096395),  INT32_C(   224130384),  INT32_C(  1295384904),  INT32_C(  1640227374), -INT32_C(  1930827861),  INT32_C(  2037067973), -INT32_C(  1886603945) },
       INT8_C(   0),
      { -INT32_C(  1639741622), -INT32_C(  1034527332),  INT32_C(   224130384),  INT32_C(  1295384904),  INT32_C(  1640227374),  INT32_C(  1024205182),  INT32_C(  2139059556),  INT32_C(   449668048),
        -INT32_C(  2034757142), -INT32_C(   604232985),  INT32_C(   490163622), -INT32_C(    97826685), -INT32_C(   604161699), -INT32_C(  2010855117),  INT32_C(  1329436799), -INT32_C(   429323269) } },
    { { -INT32_C(   727077198), -INT32_C(  1624974167),  INT32_C(   971452983),  INT32_C(  1248277608), -INT32_C(   554251785), -INT32_C(   492535118), -INT32_C(  1556537874),  INT32_C(  1556203689),
         INT32_C(   540043895),  INT32_C(  2109691206), -INT32_C(   273242489), -INT32_C(   784785702), -INT32_C(  1716506649), -INT32_C(  1602464590),  INT32_C(   440644977), -INT32_C(   176814978) },
      UINT16_C(42862),
      { -INT32_C(   704859115), -INT32_C(   394558671), -INT32_C(  1425648014), -INT32_C(   673452761),  INT32_C(    36408710),  INT32_C(  1891146797),  INT32_C(   762590903),  INT32_C(  1087693355),
        -INT32_C(   954871658), -INT32_C(   978349741),  INT32_C(   242333159),  INT32_C(   702958754),  INT32_C(   103485913),  INT32_C(  1702290350),  INT32_C(  1150478873),  INT32_C(  1652910028) },
      { -INT32_C(  1960207817),  INT32_C(   307288363),  INT32_C(   824295822), -INT32_C(   430307571), -INT32_C(   957577960), -INT32_C(  2111085975),  INT32_C(   415743564),  INT32_C(  1551518757) },
       INT8_C(   0),
      { -INT32_C(   727077198),  INT32_C(   307288363),  INT32_C(   824295822), -INT32_C(   430307571), -INT32_C(   554251785), -INT32_C(  2111085975),  INT32_C(   415743564),  INT32_C(  1556203689),
        -INT32_C(   954871658), -INT32_C(   978349741),  INT32_C(   242333159), -INT32_C(   784785702), -INT32_C(  1716506649),  INT32_C(  1702290350),  INT32_C(   440644977),  INT32_C(  1652910028) } },
    { {  INT32_C(  2098390948), -INT32_C(   116710344),  INT32_C(  1275542597),  INT32_C(   476442006),  INT32_C(  1015360474),  INT32_C(   361301847),  INT32_C(   440050126), -INT32_C(  1073637860),
        -INT32_C(  1237511298),  INT32_C(  2108639287),  INT32_C(   466204293),  INT32_C(  2117545636), -INT32_C(  1313162150), -INT32_C(  1832500284),  INT32_C(    28115173),  INT32_C(   348237462) },
      UINT16_C(65217),
      {  INT32_C(  2051471562),  INT32_C(  1060162421),  INT32_C(   510514407),  INT32_C(   232507218),  INT32_C(  1045471096), -INT32_C(   566348495), -INT32_C(   141765322), -INT32_C(  1258992151),
        -INT32_C(  1154597818), -INT32_C(   285581817), -INT32_C(  2062718925), -INT32_C(  1466767313), -INT32_C(  1192828025),  INT32_C(  1335305496), -INT32_C(   481942534), -INT32_C(  1214825359) },
      {  INT32_C(  2138228088),  INT32_C(  1466854692),  INT32_C(    81558485), -INT32_C(   357798045),  INT32_C(  1789039442), -INT32_C(  1296483912), -INT32_C(   829095843), -INT32_C(  1266340548) },
       INT8_C(   0),
      {  INT32_C(  2138228088), -INT32_C(   116710344),  INT32_C(  1275542597),  INT32_C(   476442006),  INT32_C(  1015360474),  INT32_C(   361301847), -INT32_C(   829095843), -INT32_C(  1266340548),
        -INT32_C(  1237511298), -INT32_C(   285581817), -INT32_C(  2062718925), -INT32_C(  1466767313), -INT32_C(  1192828025),  INT32_C(  1335305496), -INT32_C(   481942534), -INT32_C(  1214825359) } },
    { {  INT32_C(  1679176695),  INT32_C(   473525665), -INT32_C(  1182843318),  INT32_C(  2097900010),  INT32_C(  1161131276),  INT32_C(   799205167), -INT32_C(  1435799427), -INT32_C(   308535307),
        -INT32_C(   195972270),  INT32_C(  1779469088), -INT32_C(  1289514807),  INT32_C(    87109369), -INT32_C(   750033244), -INT32_C(   889000626),  INT32_C(  1400204638), -INT32_C(   566226548) },
      UINT16_C(37572),
      { -INT32_C(   484579886),  INT32_C(  1936975439), -INT32_C(   895390567), -INT32_C(  1120844430),  INT32_C(   447381272), -INT32_C(  1081669558), -INT32_C(  1664085156), -INT32_C(  1003579918),
        -INT32_C(   911783046), -INT32_C(   885253583), -INT32_C(   107618681),  INT32_C(  1018611235), -INT32_C(  1890164667), -INT32_C(   984686999), -INT32_C(   480174095),  INT32_C(   782799028) },
      {  INT32_C(   234311643), -INT32_C(   237489046),  INT32_C(   904555794),  INT32_C(  2037489716),  INT32_C(  1762248704), -INT32_C(  1758570586),  INT32_C(   712740982), -INT32_C(    78109920) },
       INT8_C(   0),
      {  INT32_C(  1679176695),  INT32_C(   473525665),  INT32_C(   904555794),  INT32_C(  2097900010),  INT32_C(  1161131276),  INT32_C(   799205167),  INT32_C(   712740982), -INT32_C(    78109920),
        -INT32_C(   195972270), -INT32_C(   885253583), -INT32_C(  1289514807),  INT32_C(    87109369), -INT32_C(  1890164667), -INT32_C(   889000626),  INT32_C(  1400204638),  INT32_C(   782799028) } },
    { { -INT32_C(  2065954736),  INT32_C(  1318505952),  INT32_C(  1468189623),  INT32_C(    89717565),  INT32_C(  1554759940), -INT32_C(  2133703952), -INT32_C(   526320195), -INT32_C(  1521247403),
        -INT32_C(  2077675612), -INT32_C(  1261256707), -INT32_C(   938716021),  INT32_C(  1422746704),  INT32_C(   363886629),  INT32_C(  2023064506), -INT32_C(   715639424),  INT32_C(  1971039185) },
      UINT16_C(42203),
      { -INT32_C(   865806087), -INT32_C(  1742671988), -INT32_C(  2063830857),  INT32_C(  1979458245), -INT32_C(   839338441), -INT32_C(  2029881298), -INT32_C(   919415986),  INT32_C(  1114443080),
         INT32_C(  1913573861),  INT32_C(  1997156032),  INT32_C(  1694173087),  INT32_C(  1624897321), -INT32_C(   600976722), -INT32_C(  1738330038),  INT32_C(  1281463811), -INT32_C(  1987129437) },
      {  INT32_C(  1643879584),  INT32_C(  1775764938),  INT32_C(   902681612),  INT32_C(  2039916491), -INT32_C(  1017724039), -INT32_C(   144918029), -INT32_C(   213664433),  INT32_C(   763154828) },
       INT8_C(   1),
      { -INT32_C(   865806087), -INT32_C(  1742671988),  INT32_C(  1468189623),  INT32_C(  1979458245), -INT32_C(   839338441), -INT32_C(  2133703952), -INT32_C(   919415986),  INT32_C(  1114443080),
        -INT32_C(  2077675612), -INT32_C(  1261256707),  INT32_C(   902681612),  INT32_C(  1422746704),  INT32_C(   363886629), -INT32_C(   144918029), -INT32_C(   715639424),  INT32_C(   763154828) } },
    { {  INT32_C(  2084015735),  INT32_C(   982098278),  INT32_C(   352763502),  INT32_C(   411991892), -INT32_C(  1895083307),  INT32_C(  1809711790),  INT32_C(   385405253), -INT32_C(   998038195),
         INT32_C(   423738291), -INT32_C(   900412836), -INT32_C(   572499320), -INT32_C(  1342869799),  INT32_C(  1849557184),  INT32_C(  1222253571),  INT32_C(   996135662), -INT32_C(  1442782473) },
      UINT16_C(16797),
      {  INT32_C(   386660803), -INT32_C(  1536060476),  INT32_C(  1695697776),  INT32_C(   946263034),  INT32_C(   458516801),  INT32_C(   283984561),  INT32_C(  2129912958),  INT32_C(  1388286094),
         INT32_C(  1315555978), -INT32_C(   822944931),  INT32_C(   540214310),  INT32_C(   425236951), -INT32_C(  1271616510),  INT32_C(  1841570543), -INT32_C(  1779714298), -INT32_C(   756569528) },
      { -INT32_C(   786411404),  INT32_C(  1369379371), -INT32_C(   294530282),  INT32_C(  1862781292),  INT32_C(  1696807798),  INT32_C(  1691543389), -INT32_C(   403063393), -INT32_C(   608575385) },
       INT8_C(   0),
      { -INT32_C(   786411404),  INT32_C(   982098278), -INT32_C(   294530282),  INT32_C(  1862781292),  INT32_C(  1696807798),  INT32_C(  1809711790),  INT32_C(   385405253), -INT32_C(   608575385),
         INT32_C(  1315555978), -INT32_C(   900412836), -INT32_C(   572499320), -INT32_C(  1342869799),  INT32_C(  1849557184),  INT32_C(  1222253571), -INT32_C(  1779714298), -INT32_C(  1442782473) } },
    { { -INT32_C(   346247975),  INT32_C(   520203596), -INT32_C(   393482465),  INT32_C(   845085430),  INT32_C(    93307677),  INT32_C(  1386541973), -INT32_C(   860255252),  INT32_C(   503157828),
        -INT32_C(  1928832703),  INT32_C(   632031494), -INT32_C(   284280839),  INT32_C(  1327590450), -INT32_C(   984305360), -INT32_C(  1860634460), -INT32_C(   950152829), -INT32_C(  1478206874) },
      UINT16_C(60595),
      { -INT32_C(   520701388), -INT32_C(   317198369),  INT32_C(     5917150), -INT32_C(   307131751),  INT32_C(  1743148367),  INT32_C(  1144613350),  INT32_C(   362716977),  INT32_C(  2046972486),
        -INT32_C(   346359796), -INT32_C(   975605273),  INT32_C(  1422209979),  INT32_C(   205616829), -INT32_C(  1301010484), -INT32_C(  1024021103), -INT32_C(  1831365300), -INT32_C(   217261593) },
      { -INT32_C(  1193384239), -INT32_C(  1837189161), -INT32_C(  1478081558), -INT32_C(  2035013446), -INT32_C(   533190577),  INT32_C(   564277205), -INT32_C(  1414300988),  INT32_C(   631160659) },
       INT8_C(   1),
      { -INT32_C(   520701388), -INT32_C(   317198369), -INT32_C(   393482465),  INT32_C(   845085430),  INT32_C(  1743148367),  INT32_C(  1144613350), -INT32_C(   860255252),  INT32_C(  2046972486),
        -INT32_C(  1928832703),  INT32_C(   632031494), -INT32_C(  1478081558), -INT32_C(  2035013446), -INT32_C(   984305360),  INT32_C(   564277205), -INT32_C(  1414300988),  INT32_C(   631160659) } },
    { {  INT32_C(   888987004), -INT32_C(  1625387173), -INT32_C(  1655060875), -INT32_C(  1561469062),  INT32_C(  1182256407), -INT32_C(   368404368),  INT32_C(   171816267), -INT32_C(   802201005),
        -INT32_C(  1694225600),  INT32_C(   809116346),  INT32_C(  1657639912), -INT32_C(  1979401614), -INT32_C(   120554616),  INT32_C(  1591925523), -INT32_C(   462938224), -INT32_C(  1028352126) },
      UINT16_C(47299),
      { -INT32_C(  1730511523),  INT32_C(  2066465709),  INT32_C(   708156965), -INT32_C(   123355864), -INT32_C(  1747732043), -INT32_C(  2135464937), -INT32_C(    65586872),  INT32_C(  1488247547),
         INT32_C(    99651160),  INT32_C(  2004884561), -INT32_C(   492718406),  INT32_C(   702170739),  INT32_C(   398503679),  INT32_C(  1519875858), -INT32_C(  1420382544), -INT32_C(   519828855) },
      { -INT32_C(   353962856), -INT32_C(   899586288), -INT32_C(  1867775459),  INT32_C(  1220118344),  INT32_C(  1197439541), -INT32_C(  1566443791),  INT32_C(   760084388), -INT32_C(  1727115007) },
       INT8_C(   0),
      { -INT32_C(   353962856), -INT32_C(   899586288), -INT32_C(  1655060875), -INT32_C(  1561469062),  INT32_C(  1182256407), -INT32_C(   368404368),  INT32_C(   760084388), -INT32_C(  1727115007),
        -INT32_C(  1694225600),  INT32_C(   809116346),  INT32_C(  1657639912),  INT32_C(   702170739),  INT32_C(   398503679),  INT32_C(  1519875858), -INT32_C(   462938224), -INT32_C(   519828855) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi32(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_inserti32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_epi32 ()), test_vec[i].imm8 ,src, test_vec[i].k , a, b);
    simde_assert_m512i_i32(r, == ,simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m256i b = simde_test_x86_random_i32x8();
    int imm8 = simde_test_codegen_rand() & 0x01;
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_mask_inserti32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_epi32 ()), imm8, src, k, a, b);
    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_inserti32x8 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[8];
    const int imm8;
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(27222),
      {  INT32_C(  1621519087),  INT32_C(   899885648), -INT32_C(   889970550),  INT32_C(  1880803230),  INT32_C(  1637426960),  INT32_C(   637910988), -INT32_C(   420670159), -INT32_C(   179223802),
        -INT32_C(    27854930), -INT32_C(  1405814495), -INT32_C(  1200216039),  INT32_C(     2658799), -INT32_C(  2073968200), -INT32_C(  1314232704), -INT32_C(  2103928965), -INT32_C(  1988630053) },
      {  INT32_C(    25677280), -INT32_C(   525485113), -INT32_C(   745003805),  INT32_C(  1842593973),  INT32_C(    49362049),  INT32_C(   364092314),  INT32_C(   228018994),  INT32_C(   345378612) },
       INT8_C(   0),
      {  INT32_C(           0), -INT32_C(   525485113), -INT32_C(   745003805),  INT32_C(           0),  INT32_C(    49362049),  INT32_C(           0),  INT32_C(   228018994),  INT32_C(           0),
         INT32_C(           0), -INT32_C(  1405814495),  INT32_C(           0),  INT32_C(     2658799),  INT32_C(           0), -INT32_C(  1314232704), -INT32_C(  2103928965),  INT32_C(           0) } },
    { UINT16_C( 5662),
      { -INT32_C(  2067539549), -INT32_C(  1877153859),  INT32_C(   157539740),  INT32_C(  1627035486),  INT32_C(  1175754033), -INT32_C(   706846776), -INT32_C(  1469256300),  INT32_C(  1841203913),
         INT32_C(   552698467),  INT32_C(   112201065),  INT32_C(  1225724906), -INT32_C(   609678678),  INT32_C(  1730330015), -INT32_C(  1321402339), -INT32_C(  1235637779), -INT32_C(  1776084941) },
      {  INT32_C(    62330010),  INT32_C(   201942818),  INT32_C(   626334075), -INT32_C(  1040056541), -INT32_C(   651549764),  INT32_C(   277505827),  INT32_C(  1137107984), -INT32_C(  1764103428) },
       INT8_C(   0),
      {  INT32_C(           0),  INT32_C(   201942818),  INT32_C(   626334075), -INT32_C(  1040056541), -INT32_C(   651549764),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(   112201065),  INT32_C(  1225724906),  INT32_C(           0),  INT32_C(  1730330015),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(39312),
      {  INT32_C(   765720608), -INT32_C(  1736262541), -INT32_C(  1583775265),  INT32_C(   399227966), -INT32_C(   274582817), -INT32_C(  2051635646),  INT32_C(   408920194), -INT32_C(  1078857826),
         INT32_C(  1525437671), -INT32_C(   269259248),  INT32_C(   781290736),  INT32_C(   675634249), -INT32_C(   786897009), -INT32_C(   279458195),  INT32_C(   201832046), -INT32_C(  1916028507) },
      {  INT32_C(   518502157),  INT32_C(   370006566), -INT32_C(  1337680281), -INT32_C(  1965520389), -INT32_C(   564400015),  INT32_C(   751743678),  INT32_C(   238671208), -INT32_C(  1667562354) },
       INT8_C(   0),
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   564400015),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1667562354),
         INT32_C(  1525437671),  INT32_C(           0),  INT32_C(           0),  INT32_C(   675634249), -INT32_C(   786897009),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1916028507) } },
    { UINT16_C(47746),
      { -INT32_C(   121152030),  INT32_C(  1950115524), -INT32_C(   347290271),  INT32_C(   356924726),  INT32_C(   686029307),  INT32_C(  1868675169), -INT32_C(   485857977),  INT32_C(    60656929),
        -INT32_C(  1359256342),  INT32_C(   723662794),  INT32_C(   857108221), -INT32_C(  1488429653), -INT32_C(  1211159722),  INT32_C(   723988708), -INT32_C(  1223806571), -INT32_C(  1464161346) },
      { -INT32_C(   631851505), -INT32_C(   368740115), -INT32_C(  1843520281), -INT32_C(   818321799),  INT32_C(  1988495506), -INT32_C(   828265160), -INT32_C(  1652183073),  INT32_C(  1766211418) },
       INT8_C(   1),
      {  INT32_C(           0),  INT32_C(  1950115524),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(    60656929),
         INT32_C(           0), -INT32_C(   368740115),  INT32_C(           0), -INT32_C(   818321799),  INT32_C(  1988495506), -INT32_C(   828265160),  INT32_C(           0),  INT32_C(  1766211418) } },
    { UINT16_C(17308),
      { -INT32_C(   850848285), -INT32_C(  1897175556), -INT32_C(  1379380514),  INT32_C(  1513345252), -INT32_C(   671358711),  INT32_C(  1566354112),  INT32_C(  1839438596),  INT32_C(  1957773457),
         INT32_C(  1363278421),  INT32_C(  1038036319),  INT32_C(  1676322687), -INT32_C(  2118312584), -INT32_C(  1101481730),  INT32_C(  1713157218), -INT32_C(   522993841),  INT32_C(  1414825215) },
      { -INT32_C(   576350594),  INT32_C(  1109034435), -INT32_C(  1532689364),  INT32_C(   539320866),  INT32_C(  2095021338), -INT32_C(  2115830991), -INT32_C(  1168001606), -INT32_C(  1206995398) },
       INT8_C(   0),
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1532689364),  INT32_C(   539320866),  INT32_C(  2095021338),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1206995398),
         INT32_C(  1363278421),  INT32_C(  1038036319),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   522993841),  INT32_C(           0) } },
    { UINT16_C(38580),
      {  INT32_C(  1370503439),  INT32_C(   183940453), -INT32_C(   147892009), -INT32_C(   287920782),  INT32_C(  1607520734),  INT32_C(  1187087756),  INT32_C(  2018801599), -INT32_C(   770832189),
        -INT32_C(  1490829759),  INT32_C(  1253120371), -INT32_C(   482156431),  INT32_C(  1825642637),  INT32_C(  1993056746), -INT32_C(   423850970), -INT32_C(   966914045),  INT32_C(  1519938585) },
      { -INT32_C(  1644053718),  INT32_C(  1172878036),  INT32_C(   539503251),  INT32_C(   764213571), -INT32_C(  1046259558), -INT32_C(   391684124), -INT32_C(  2001861265), -INT32_C(  1679603855) },
       INT8_C(   0),
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   539503251),  INT32_C(           0), -INT32_C(  1046259558), -INT32_C(   391684124),  INT32_C(           0), -INT32_C(  1679603855),
         INT32_C(           0),  INT32_C(  1253120371), -INT32_C(   482156431),  INT32_C(           0),  INT32_C(  1993056746),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1519938585) } },
    { UINT16_C(14820),
      {  INT32_C(   472029143),  INT32_C(  1246055466), -INT32_C(  1126744433), -INT32_C(  1705037863), -INT32_C(    79577581), -INT32_C(  1230354898),  INT32_C(  1385820599), -INT32_C(   896762125),
         INT32_C(  1072147989), -INT32_C(  1970721542),  INT32_C(  1128685674), -INT32_C(  1545754993), -INT32_C(  1835131292),  INT32_C(   457787492),  INT32_C(   762241593),  INT32_C(  1979185760) },
      { -INT32_C(  1565204824),  INT32_C(  1965833738),  INT32_C(   783840158),  INT32_C(  2110887449),  INT32_C(   420507572), -INT32_C(   248227400), -INT32_C(  1675713989),  INT32_C(  1141970588) },
       INT8_C(   0),
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   783840158),  INT32_C(           0),  INT32_C(           0), -INT32_C(   248227400), -INT32_C(  1675713989),  INT32_C(  1141970588),
         INT32_C(  1072147989),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1545754993), -INT32_C(  1835131292),  INT32_C(   457787492),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(59334),
      {  INT32_C(  1947403519), -INT32_C(   802388318),  INT32_C(   497140383), -INT32_C(  1876094601), -INT32_C(  1161525559),  INT32_C(  1574528961),  INT32_C(  1215295235), -INT32_C(   500222493),
        -INT32_C(   615103943),  INT32_C(  1756136393), -INT32_C(  1132114619),  INT32_C(   659337822), -INT32_C(    85847752),  INT32_C(  2069347192), -INT32_C(  1916549462),  INT32_C(   879817467) },
      { -INT32_C(    32455115), -INT32_C(  1889092535),  INT32_C(  1749806089), -INT32_C(   678455138),  INT32_C(   567374249), -INT32_C(   694409172), -INT32_C(   362520337), -INT32_C(  2027957166) },
       INT8_C(   0),
      {  INT32_C(           0), -INT32_C(  1889092535),  INT32_C(  1749806089),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   362520337), -INT32_C(  2027957166),
        -INT32_C(   615103943),  INT32_C(  1756136393), -INT32_C(  1132114619),  INT32_C(           0),  INT32_C(           0),  INT32_C(  2069347192), -INT32_C(  1916549462),  INT32_C(   879817467) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi32(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_inserti32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_epi32 ()), test_vec[i].imm8 , test_vec[i].k , a, b);
    simde_assert_m512i_i32(r, == ,simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m256i b = simde_test_x86_random_i32x8();
    int imm8 = simde_test_codegen_rand() & 0x01;
    simde__m512i r;
    SIMDE_CONSTIFY_2_(simde_mm512_maskz_inserti32x8, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_epi32 ()), imm8, k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_inserti64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[8];
    const int64_t b[2];
    const int imm8;
    const int64_t r[8];
  } test_vec[] = {
    { { -INT64_C( 3185839351605653724), -INT64_C( 7266773486870972167), -INT64_C( 7447103374582978907), -INT64_C( 4936552570322563634),
         INT64_C( 3113875420402652006), -INT64_C( 1474061058599679251),  INT64_C( 5333249854633994620),  INT64_C( 2423611407202837583) },
      {  INT64_C( 3483176159473340152), -INT64_C( 2957268077581875772) },
       INT8_C(   2),
      { -INT64_C( 3185839351605653724), -INT64_C( 7266773486870972167), -INT64_C( 7447103374582978907), -INT64_C( 4936552570322563634),
         INT64_C( 3483176159473340152), -INT64_C( 2957268077581875772),  INT64_C( 5333249854633994620),  INT64_C( 2423611407202837583) } },
    { {  INT64_C( 2928285631258968529), -INT64_C( 4926370448219600316),  INT64_C( 7495729358787497352), -INT64_C( 2913097753665793969),
         INT64_C( 4920410192332170459), -INT64_C( 5169693458659726503), -INT64_C( 4762555031822319780),  INT64_C( 2373198288787188498) },
      {  INT64_C( 4842363467459402735),  INT64_C(  980390198247039411) },
       INT8_C(   3),
      {  INT64_C( 2928285631258968529), -INT64_C( 4926370448219600316),  INT64_C( 7495729358787497352), -INT64_C( 2913097753665793969),
         INT64_C( 4920410192332170459), -INT64_C( 5169693458659726503),  INT64_C( 4842363467459402735),  INT64_C(  980390198247039411) } },
    { {  INT64_C( 2425881596737981846), -INT64_C( 3716728970626547152), -INT64_C( 5836389627976066363),  INT64_C( 1520647116091810756),
         INT64_C( 8831654846486402041),  INT64_C( 4726352423045953049),  INT64_C(  862595765469334971), -INT64_C( 4826181199454316687) },
      {  INT64_C( 2489852916168176329), -INT64_C( 7029974006557277518) },
       INT8_C(   1),
      {  INT64_C( 2425881596737981846), -INT64_C( 3716728970626547152),  INT64_C( 2489852916168176329), -INT64_C( 7029974006557277518),
         INT64_C( 8831654846486402041),  INT64_C( 4726352423045953049),  INT64_C(  862595765469334971), -INT64_C( 4826181199454316687) } },
    { {  INT64_C( 3962645264696141822), -INT64_C( 6503838928926478593), -INT64_C( 2282251485583585052), -INT64_C( 1148028736906231107),
         INT64_C( 5574748588866349640), -INT64_C(  863566257759102870),  INT64_C( 4543544344406504143),  INT64_C( 8740802093004148843) },
      { -INT64_C( 5668064599453152325),  INT64_C( 7375228880729118805) },
       INT8_C(   1),
      {  INT64_C( 3962645264696141822), -INT64_C( 6503838928926478593), -INT64_C( 5668064599453152325),  INT64_C( 7375228880729118805),
         INT64_C( 5574748588866349640), -INT64_C(  863566257759102870),  INT64_C( 4543544344406504143),  INT64_C( 8740802093004148843) } },
    { {  INT64_C( 8089458480580629487), -INT64_C( 5092415666583920633),  INT64_C( 5480231711319023034), -INT64_C( 4856769574753569855),
         INT64_C( 7993564327629414621),  INT64_C( 7214867390478651257), -INT64_C( 7874496869506998088), -INT64_C( 4456593420438824035) },
      {  INT64_C(  988027283128171488), -INT64_C( 3401462261012447811) },
       INT8_C(   0),
      {  INT64_C(  988027283128171488), -INT64_C( 3401462261012447811),  INT64_C( 5480231711319023034), -INT64_C( 4856769574753569855),
         INT64_C( 7993564327629414621),  INT64_C( 7214867390478651257), -INT64_C( 7874496869506998088), -INT64_C( 4456593420438824035) } },
    { { -INT64_C( 7949107376428859928), -INT64_C(  333767477023073922),  INT64_C( 1757138722061526391),  INT64_C( 8959913196575050000),
         INT64_C( 8413918127176211774), -INT64_C( 1544364744122418602), -INT64_C( 3717174226293153547), -INT64_C( 2752886177845437766) },
      { -INT64_C( 1364641712998426203),  INT64_C( 5073327320480284939) },
       INT8_C(   0),
      { -INT64_C( 1364641712998426203),  INT64_C( 5073327320480284939),  INT64_C( 1757138722061526391),  INT64_C( 8959913196575050000),
         INT64_C( 8413918127176211774), -INT64_C( 1544364744122418602), -INT64_C( 3717174226293153547), -INT64_C( 2752886177845437766) } },
    { { -INT64_C( 7232011335214524425),  INT64_C( 1556573921140469804),  INT64_C( 5041948304148804891), -INT64_C( 3069523249133619683),
         INT64_C( 4544236561201190516),  INT64_C( 8079684596158187087), -INT64_C( 9039756458562040829),  INT64_C( 8250061248460497611) },
      { -INT64_C( 1602164799403771024), -INT64_C( 4034358002387192003) },
       INT8_C(   0),
      { -INT64_C( 1602164799403771024), -INT64_C( 4034358002387192003),  INT64_C( 5041948304148804891), -INT64_C( 3069523249133619683),
         INT64_C( 4544236561201190516),  INT64_C( 8079684596158187087), -INT64_C( 9039756458562040829),  INT64_C( 8250061248460497611) } },
    { { -INT64_C( 4791609432435454112), -INT64_C( 2159706198977599108),  INT64_C( 4920856924468840981),  INT64_C( 4634153303436549466),
        -INT64_C( 2722296930373364256),  INT64_C( 3974276185255376030),  INT64_C( 7103459778122523394),  INT64_C( 5731992088138063212) },
      { -INT64_C( 7715787706180337227), -INT64_C( 3003618565512667657) },
       INT8_C(   3),
      { -INT64_C( 4791609432435454112), -INT64_C( 2159706198977599108),  INT64_C( 4920856924468840981),  INT64_C( 4634153303436549466),
        -INT64_C( 2722296930373364256),  INT64_C( 3974276185255376030), -INT64_C( 7715787706180337227), -INT64_C( 3003618565512667657) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi64(test_vec[i].b);
    switch(test_vec[i].imm8) {
      case 0: simde_assert_m512i_i64(simde_mm512_inserti64x2(a, b, 0), ==, simde_mm512_loadu_epi64(test_vec[i].r)); break;
      case 1: simde_assert_m512i_i64(simde_mm512_inserti64x2(a, b, 1), ==, simde_mm512_loadu_epi64(test_vec[i].r)); break;
      case 2: simde_assert_m512i_i64(simde_mm512_inserti64x2(a, b, 2), ==, simde_mm512_loadu_epi64(test_vec[i].r)); break;
      case 3: simde_assert_m512i_i64(simde_mm512_inserti64x2(a, b, 3), ==, simde_mm512_loadu_epi64(test_vec[i].r)); break;
    }
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m128i b = simde_test_x86_random_i64x2();
    int imm8 = simde_test_codegen_rand() & 0x03;
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_inserti64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, a, b);

    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_inserti64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[2];
    const int imm8;
    const int64_t r[8];
  } test_vec[] = {
    { { -INT64_C( 1437525857059302259),  INT64_C( 8028295443111167418), -INT64_C( 4479432208217808138), -INT64_C( 5374044330178084993),
        -INT64_C( 7932756663924951773), -INT64_C( 6697009134660259724),  INT64_C( 2681675490236090963),  INT64_C( 8256546512561618872) },
      UINT8_C( 26),
      { -INT64_C( 3937410130845274184), -INT64_C( 3148834129031332983), -INT64_C( 5084668455614168195), -INT64_C( 7162558610553680038),
        -INT64_C( 7562988475700245527),  INT64_C( 1565246718257234137),  INT64_C( 3266864359775417250), -INT64_C( 4536735779238682998) },
      {  INT64_C( 5004921317829610961), -INT64_C( 8630056981838166131) },
       INT8_C(   3),
      { -INT64_C( 1437525857059302259), -INT64_C( 3148834129031332983), -INT64_C( 4479432208217808138), -INT64_C( 7162558610553680038),
        -INT64_C( 7562988475700245527), -INT64_C( 6697009134660259724),  INT64_C( 2681675490236090963),  INT64_C( 8256546512561618872) } },
    { { -INT64_C( 2680745154261342969), -INT64_C( 3325780489277326655),  INT64_C( 3680146973240702583),  INT64_C( 8907734634907383978),
         INT64_C( 6565556226942207431), -INT64_C( 4838209306494840635),  INT64_C( 4922245938614966512), -INT64_C( 3351683033428170612) },
      UINT8_C(243),
      {  INT64_C(  182932560333837370), -INT64_C( 8758090127151492278),  INT64_C( 5270729859006501189), -INT64_C( 2552093332908857804),
         INT64_C( 6227398613268439864), -INT64_C(  310379255473458243),  INT64_C( 7569348338786816481), -INT64_C( 4873280160573726752) },
      { -INT64_C( 1934979760913973564),  INT64_C( 3528501382326779973) },
       INT8_C(   2),
      {  INT64_C(  182932560333837370), -INT64_C( 8758090127151492278),  INT64_C( 3680146973240702583),  INT64_C( 8907734634907383978),
        -INT64_C( 1934979760913973564),  INT64_C( 3528501382326779973),  INT64_C( 7569348338786816481), -INT64_C( 4873280160573726752) } },
    { {  INT64_C( 6711825787827843723),  INT64_C(  334000871406872381), -INT64_C( 7522961261384142398),  INT64_C( 1690564604027376930),
        -INT64_C( 5952489389805453815), -INT64_C( 6614892982488578774), -INT64_C( 7724638241617846114), -INT64_C( 3185470132537740505) },
      UINT8_C( 45),
      { -INT64_C( 8875387328334537945), -INT64_C( 7520718671352298539),  INT64_C( 6249762233795597826),  INT64_C( 6931870202701311510),
        -INT64_C( 1111581954520985394),  INT64_C(  409989203325286626),  INT64_C( 7810180006874298730),  INT64_C( 4998331992399512515) },
      {  INT64_C( 2918021699780962164),  INT64_C( 7991058117083361758) },
       INT8_C(   2),
      { -INT64_C( 8875387328334537945),  INT64_C(  334000871406872381),  INT64_C( 6249762233795597826),  INT64_C( 6931870202701311510),
        -INT64_C( 5952489389805453815),  INT64_C( 7991058117083361758), -INT64_C( 7724638241617846114), -INT64_C( 3185470132537740505) } },
    { {  INT64_C( 7032210146969287551),  INT64_C( 6410344042728035550),  INT64_C( 8373253849835537776),  INT64_C( 8692765291580036103),
        -INT64_C( 1250905513012714653),  INT64_C(  630899327110314872),  INT64_C( 3342956419843186924),  INT64_C( 2888999536580550053) },
      UINT8_C(108),
      { -INT64_C( 8665090557066057373),  INT64_C( 5345535169814014171),  INT64_C( 7042396732473220997), -INT64_C( 3100079148162983483),
        -INT64_C( 4878282537939798344),  INT64_C( 1744914593517366019),  INT64_C(  807245248612706382),  INT64_C( 6955690502273980994) },
      { -INT64_C( 9104464811069693431),  INT64_C( 2416083152145393116) },
       INT8_C(   2),
      {  INT64_C( 7032210146969287551),  INT64_C( 6410344042728035550),  INT64_C( 7042396732473220997), -INT64_C( 3100079148162983483),
        -INT64_C( 1250905513012714653),  INT64_C( 2416083152145393116),  INT64_C(  807245248612706382),  INT64_C( 2888999536580550053) } },
    { { -INT64_C( 9087806481032219723), -INT64_C( 4767471303514287047),  INT64_C(  863772375361765551), -INT64_C( 6260565384817054678),
        -INT64_C( 8314726474627697676), -INT64_C(  102122297072433732), -INT64_C( 5304647121518014224),  INT64_C( 3901143407175090567) },
      UINT8_C( 88),
      { -INT64_C( 5597911743785643669), -INT64_C( 8308577551853148271), -INT64_C( 5926227575199758049), -INT64_C( 3867425085825533098),
        -INT64_C( 3957066244173201546),  INT64_C( 3971677997605984213),  INT64_C( 6583754705295455813), -INT64_C( 1686972500860777107) },
      {  INT64_C( 8551034565895648426), -INT64_C( 5204148570918734472) },
       INT8_C(   0),
      { -INT64_C( 9087806481032219723), -INT64_C( 4767471303514287047),  INT64_C(  863772375361765551), -INT64_C( 3867425085825533098),
        -INT64_C( 3957066244173201546), -INT64_C(  102122297072433732),  INT64_C( 6583754705295455813),  INT64_C( 3901143407175090567) } },
    { {  INT64_C( 1429820066599430200),  INT64_C( 3344605772364751892),  INT64_C( 3211426036395822264),  INT64_C( 2364766835735142734),
         INT64_C(  403179852310131646),  INT64_C( 4930929817639198213), -INT64_C( 3197856428486476166),  INT64_C( 5843167160541133839) },
      UINT8_C( 76),
      { -INT64_C( 8356357167846205791), -INT64_C(  743273475865032698), -INT64_C(  308974043179376455),  INT64_C( 7348430678331862421),
         INT64_C( 9128637531484010420), -INT64_C( 2566099458350397513), -INT64_C( 4594734666174177918),  INT64_C( 6633636949274839526) },
      { -INT64_C( 7019088828933638240),  INT64_C( 3384014954977577718) },
       INT8_C(   0),
      {  INT64_C( 1429820066599430200),  INT64_C( 3344605772364751892), -INT64_C(  308974043179376455),  INT64_C( 7348430678331862421),
         INT64_C(  403179852310131646),  INT64_C( 4930929817639198213), -INT64_C( 4594734666174177918),  INT64_C( 5843167160541133839) } },
    { {  INT64_C( 8474145759929200506), -INT64_C( 3769942436141513563), -INT64_C( 6404594358540974343), -INT64_C( 6262675090042570661),
         INT64_C( 7499327649544441329), -INT64_C( 6820098379319201989), -INT64_C( 5264452485635306517), -INT64_C( 5190231696846273760) },
      UINT8_C( 36),
      { -INT64_C( 7672356514888657835),  INT64_C( 8923565383462229447),  INT64_C( 5958697780470226856),  INT64_C( 5677297476400840717),
        -INT64_C( 3954688227514142429), -INT64_C( 2080509943840744651), -INT64_C( 8067314163628378454),  INT64_C( 1543889088123577125) },
      {  INT64_C( 6592069349683365296),  INT64_C( 7220713562170208024) },
       INT8_C(   2),
      {  INT64_C( 8474145759929200506), -INT64_C( 3769942436141513563),  INT64_C( 5958697780470226856), -INT64_C( 6262675090042570661),
         INT64_C( 7499327649544441329),  INT64_C( 7220713562170208024), -INT64_C( 5264452485635306517), -INT64_C( 5190231696846273760) } },
    { {  INT64_C( 5544538251254570409), -INT64_C( 9075786255958077347),  INT64_C( 7550005747093323507), -INT64_C(  560549379662734846),
         INT64_C( 6276532413749711269), -INT64_C( 6789247158023136823),  INT64_C( 8367152034370056061),  INT64_C( 7960903755700257074) },
      UINT8_C(160),
      { -INT64_C( 5710018596200432576),  INT64_C( 4458563417998755747), -INT64_C(  749528605239831217), -INT64_C( 3087168523806565447),
         INT64_C( 6228354807561568872),  INT64_C( 1991321988881763734),  INT64_C( 3303140140024811821),  INT64_C( 3071981349082928439) },
      { -INT64_C( 7734553035679371360),  INT64_C( 2814075740473549093) },
       INT8_C(   3),
      {  INT64_C( 5544538251254570409), -INT64_C( 9075786255958077347),  INT64_C( 7550005747093323507), -INT64_C(  560549379662734846),
         INT64_C( 6276532413749711269),  INT64_C( 1991321988881763734),  INT64_C( 8367152034370056061),  INT64_C( 2814075740473549093) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_inserti64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), test_vec[i].imm8, src, test_vec[i].k, a, b);
    simde_assert_m512i_i64(r, ==, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i64x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m128i b = simde_test_x86_random_i64x2();
    int imm8 = simde_test_codegen_rand() & 0x03;
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_mask_inserti64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, src, k, a, b);

    simde_test_x86_write_i64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_inserti64x2 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[2];
    const int imm8;
    const int64_t r[8];
  } test_vec[] = {
    { UINT8_C(202),
      { -INT64_C( 8237243270128445417),  INT64_C( 8476585602763714698),  INT64_C( 9036907395087082997),  INT64_C( 7218088560004376325),
        -INT64_C( 6298971144647037147),  INT64_C( 6315501753178030463),  INT64_C( 8733377380225640784), -INT64_C( 8860544883512143273) },
      { -INT64_C(  541800324037482060),  INT64_C( 9177718312718107556) },
       INT8_C(   0),
      {  INT64_C(                   0),  INT64_C( 9177718312718107556),  INT64_C(                   0),  INT64_C( 7218088560004376325),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 8733377380225640784), -INT64_C( 8860544883512143273) } },
    { UINT8_C( 36),
      { -INT64_C( 9152813461978928076),  INT64_C( 7801424081693530449), -INT64_C( 2339000197414142708),  INT64_C( 7834685795484940601),
         INT64_C(  988767426010419090), -INT64_C( 9046977979008664281),  INT64_C( 7926216820792033642), -INT64_C( 6561255294863430323) },
      { -INT64_C( 4929589380716378675),  INT64_C( 1642181829178032081) },
       INT8_C(   0),
      {  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 2339000197414142708),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 9046977979008664281),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(212),
      { -INT64_C( 2818209014458324937),  INT64_C( 8642797574196426298),  INT64_C(   36553536830145771),  INT64_C( 8846027634559364389),
         INT64_C( 1908564034654907343),  INT64_C( 6961500095989027466), -INT64_C( 6198251039332587589),  INT64_C( 7371931442916962192) },
      {  INT64_C( 2198750110262427528), -INT64_C( 4941048024840025703) },
       INT8_C(   2),
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(   36553536830145771),  INT64_C(                   0),
         INT64_C( 2198750110262427528),  INT64_C(                   0), -INT64_C( 6198251039332587589),  INT64_C( 7371931442916962192) } },
    { UINT8_C( 79),
      {  INT64_C( 5611839954539908022), -INT64_C( 1701460307757861098),  INT64_C( 6333758808429426330),  INT64_C( 6400386526130660885),
        -INT64_C( 2464898604019841592),  INT64_C( 4623126051445372003), -INT64_C( 5402273775374919781), -INT64_C( 4457144183912121045) },
      {  INT64_C( 9093903560600400015), -INT64_C( 5376199430695324035) },
       INT8_C(   3),
      {  INT64_C( 5611839954539908022), -INT64_C( 1701460307757861098),  INT64_C( 6333758808429426330),  INT64_C( 6400386526130660885),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 9093903560600400015),  INT64_C(                   0) } },
    { UINT8_C(193),
      {  INT64_C(  840257589601508965), -INT64_C( 6533696776290523906), -INT64_C( 6005783280334508036), -INT64_C( 2561682358225551026),
        -INT64_C( 6336298963174917839), -INT64_C( 3989872829598521707), -INT64_C( 3879674626741546866),  INT64_C(  123268240808655970) },
      { -INT64_C( 8882199153304208023),  INT64_C( 3949975349483309117) },
       INT8_C(   1),
      {  INT64_C(  840257589601508965),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C(                   0), -INT64_C( 3879674626741546866),  INT64_C(  123268240808655970) } },
    { UINT8_C(203),
      {  INT64_C( 3717666206944849330),  INT64_C( 7879200824111556262),  INT64_C( 5468427451715831711),  INT64_C( 8243521086770940618),
         INT64_C( 5095229927176655563),  INT64_C(  153116218459543370),  INT64_C( 3339208447166882625),  INT64_C( 7853486795597202378) },
      {  INT64_C( 8784450493883501058), -INT64_C( 4260323317263922821) },
       INT8_C(   2),
      {  INT64_C( 3717666206944849330),  INT64_C( 7879200824111556262),  INT64_C(                   0),  INT64_C( 8243521086770940618),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 3339208447166882625),  INT64_C( 7853486795597202378) } },
    { UINT8_C( 61),
      {  INT64_C( 5470544792410477264), -INT64_C( 1793981890144006958), -INT64_C( 1328272984660817632), -INT64_C( 6196629995055016476),
        -INT64_C( 7474641923042345352),  INT64_C( 6713039919191211849), -INT64_C( 7027031887950007093),  INT64_C( 5732910629583499788) },
      {  INT64_C( 7367761574389899745), -INT64_C(  727381738952022196) },
       INT8_C(   2),
      {  INT64_C( 5470544792410477264),  INT64_C(                   0), -INT64_C( 1328272984660817632), -INT64_C( 6196629995055016476),
         INT64_C( 7367761574389899745), -INT64_C(  727381738952022196),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 76),
      {  INT64_C( 4805147150872953218),  INT64_C( 2172291751768065589),  INT64_C( 5914624517705777966), -INT64_C( 5353466923581162112),
         INT64_C( 8938907796388378603),  INT64_C( 7483969078957224679), -INT64_C(   97697662642124887),  INT64_C( 6777102096969156646) },
      {  INT64_C(  636192959912066833),  INT64_C( 2633094482387867113) },
       INT8_C(   0),
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 5914624517705777966), -INT64_C( 5353466923581162112),
         INT64_C(                   0),  INT64_C(                   0), -INT64_C(   97697662642124887),  INT64_C(                   0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_inserti64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), test_vec[i].imm8, test_vec[i].k, a, b);
    simde_assert_m512i_i64(r, ==, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m128i b = simde_test_x86_random_i64x2();
    int imm8 = simde_test_codegen_rand() & 0x03;
    simde__m512i r;
    SIMDE_CONSTIFY_4_(simde_mm512_maskz_inserti64x2, r, (HEDLEY_UNREACHABLE(), simde_mm512_setzero_si512 ()), imm8, k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_codegen_write_i8(2, HEDLEY_STATIC_CAST(int8_t, imm8), SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_insertf32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_insertf64x2)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_inserti32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_inserti64x2)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_insertf32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_insertf32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_insertf32x4)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_insertf64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_insertf64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_insertf64x4)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_inserti32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_inserti32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_inserti32x4)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_inserti64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_inserti64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_inserti64x4)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_insertf32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_insertf32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_insertf32x8)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_insertf64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_insertf64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_insertf64x2)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_inserti32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_inserti32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_inserti32x8)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_inserti64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_inserti64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_inserti64x2)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
