//@HEADER
// ***********************************************************************
//
//     EpetraExt: Epetra Extended - Linear Algebra Services Package
//                 Copyright (2011) Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Michael A. Heroux (maherou@sandia.gov)
//
// ***********************************************************************
//@HEADER

#include <EpetraExt_Reindex_LinearProblem2.h>

#include <EpetraExt_Reindex_CrsMatrix.h>
#include <EpetraExt_Reindex_MultiVector.h>

#include <Epetra_LinearProblem.h>
#include <Epetra_CrsMatrix.h>
#include <Epetra_MultiVector.h>
#include <Epetra_Map.h>

namespace EpetraExt {

LinearProblem_Reindex2::
~LinearProblem_Reindex2()
{
  if( newObj_ ) delete newObj_;

  if( MatTrans_ ) delete MatTrans_;
  if( LHSTrans_ ) delete LHSTrans_;
  if( RHSTrans_ ) delete RHSTrans_;

  if( NewRowMapOwned_ ) delete NewRowMap_;
}

LinearProblem_Reindex2::NewTypeRef
LinearProblem_Reindex2::
operator()( OriginalTypeRef orig )
{
  // Save original object
  origObj_ = &orig;

  Epetra_CrsMatrix * OldMatrix = dynamic_cast<Epetra_CrsMatrix*>( orig.GetMatrix() );
  Epetra_MultiVector * OldRHS = orig.GetRHS();
  Epetra_MultiVector * OldLHS = orig.GetLHS();
  const Epetra_BlockMap & OldRowMap = OldMatrix->Map();

  //If new map not passed in create one with lex ordering
  if( !NewRowMap_ )
  {
    int NumMyElements = OldRowMap.NumMyElements();

#ifndef EPETRA_NO_32BIT_GLOBAL_INDICES
    if(OldRowMap.GlobalIndicesInt()) {
      int NumGlobalElements = OldRowMap.NumGlobalElements();
      NewRowMap_ = new Epetra_Map( NumGlobalElements, NumMyElements, 0, OldRowMap.Comm() );
    }
    else
#endif
#ifndef EPETRA_NO_64BIT_GLOBAL_INDICES
    if(OldRowMap.GlobalIndicesLongLong()) {
      long long NumGlobalElements = OldRowMap.NumGlobalElements64();
      NewRowMap_ = new Epetra_Map( NumGlobalElements, NumMyElements, 0LL, OldRowMap.Comm() );
    }
    else
#endif
      throw "LinearProblem_Reindex2::operator(): GlobalIndices type unknown";
    NewRowMapOwned_ = true;
  }

  MatTrans_ = new CrsMatrix_Reindex( *NewRowMap_ );
  LHSTrans_ = new MultiVector_Reindex( *NewRowMap_ );
  RHSTrans_ = new MultiVector_Reindex( *NewRowMap_ );

  Epetra_CrsMatrix * NewMatrix = &((*MatTrans_)( *OldMatrix ));
  Epetra_MultiVector * NewLHS = &((*LHSTrans_)( *OldLHS ));
  Epetra_MultiVector * NewRHS = &((*RHSTrans_)( *OldRHS ));

  newObj_ = new Epetra_LinearProblem( NewMatrix, NewLHS, NewRHS );

  return *newObj_;
}

bool LinearProblem_Reindex2::fwd()
{
  Epetra_MultiVector * OldRHS = origObj_->GetRHS();
  Epetra_MultiVector * OldLHS = origObj_->GetLHS();

  // Clean up x and b before creating new multivectors
  delete (newObj_->GetLHS());
  delete (newObj_->GetRHS());

  // Create new multivectors that view into the current LHS / RHS
  Epetra_MultiVector * NewLHS = &((*LHSTrans_)( *OldLHS ));
  Epetra_MultiVector * NewRHS = &((*RHSTrans_)( *OldRHS ));

  // Set the new LHS / RHS
  newObj_->SetLHS( NewLHS );
  newObj_->SetRHS( NewRHS );

  return true;
}

} //namespace EpetraExt

