# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Test utilities for assets."""

from pgpy import PGPKey, PGPKeyring
from pgpy.constants import EllipticCurveOID, PubKeyAlgorithm

from debusine.assets.asset_utils import ExportablePGPKeyring
from debusine.test.base import TestCase


class ExportablePGPKeyringTests(TestCase):
    """Test exporting OpenPGP keyrings."""

    def test_single_key(self) -> None:
        key = PGPKey.new(PubKeyAlgorithm.EdDSA, EllipticCurveOID.Ed25519).pubkey
        keyring_armor = str(ExportablePGPKeyring(key))
        self.assertTrue(
            keyring_armor.startswith("-----BEGIN PGP PUBLIC KEY BLOCK-----\n")
        )
        self.assertEqual(keyring_armor, str(key))

    def test_multiple_keys(self) -> None:
        keys = [
            PGPKey.new(PubKeyAlgorithm.EdDSA, EllipticCurveOID.Ed25519).pubkey
            for _ in range(2)
        ]
        keyring_armor = str(ExportablePGPKeyring(*keys))
        self.assertTrue(
            keyring_armor.startswith("-----BEGIN PGP PUBLIC KEY BLOCK-----\n")
        )
        self.assertEqual(
            PGPKeyring(keyring_armor).fingerprints(),
            {key.fingerprint for key in keys},
        )
